<?php

namespace Modules\ZimflowApi\Services;
use Illuminate\Support\Facades\Config; // Good practice to import

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class ZimflowService
{
    protected string $apiEndpoint;
    protected string $apiToken;

    public function __construct()
    {
        $this->apiEndpoint = rtrim(Config::get('ZimflowApi.api_endpoint'), '/');
        $this->apiToken = Config::get('ZimflowApi.api_token');

        if (empty($this->apiToken)) {
            Log::error('Zimflow API token is not configured.');
            // throw new \RuntimeException('Zimflow API token is not configured.');
        }
        if (empty($this->apiEndpoint) || $this->apiEndpoint === '/') {
            Log::error('Zimflow API endpoint is not configured properly.');
            // throw new \RuntimeException('Zimflow API endpoint is not configured properly.');
        }
    }

    /**
     * Make contact with a user via Zimflow.
     *
     * @param string $phone The phone number of the contact.
     * @param string|null $name The name of the contact (optional).
     * @return array|null The API response or null on failure.
     */
    public function makeContact(string $phone, ?string $name = null): ?array
    {
        if (empty($this->apiToken) || empty($this->apiEndpoint) || $this->apiEndpoint === '/') {
            Log::warning('ZimflowService::makeContact called but service is not configured.');
            return ['status' => 'error', 'message' => 'Zimflow service (token/endpoint) not configured.'];
        }

        $payload = [
            'token' => $this->apiToken,
            'phone' => $phone,
        ];

        if ($name) {
            $payload['name'] = $name;
        }

        try {
            $response = Http::post("{$this->apiEndpoint}/api/wpbox/makeContact", $payload);

            if ($response->successful()) {
                return $response->json();
            }

            Log::error('Zimflow MakeContact API error', [
                'status' => $response->status(),
                'body' => $response->body(),
                'payload' => array_merge($payload, ['token' => '***MASKED***'])
            ]);
            return ['status' => 'error', 'message' => 'API request failed.', 'details' => $response->json()];

        } catch (\Exception $e) {
            Log::error('Zimflow MakeContact exception: ' . $e->getMessage(), ['payload' => array_merge($payload, ['token' => '***MASKED***'])]);
            return ['status' => 'error', 'message' => $e->getMessage()];
        }
    }

    /**
     * Make a reservation reminder via Zimflow.
     *
     * @param string $phone User's phone number.
     * @param string $name User's name.
     * @param string $source Source of the reservation.
     * @param string $startDate Reservation start date (Y-m-d H:i:s).
     * @param string $endDate Reservation end date (Y-m-d H:i:s).
     * @param string $externalId External ID for the reservation.
     * @return array|null The API response or null on failure.
     */
    public function sendReservationReminder(
        string $phone,
        string $name,
        string $source,
        string $startDate,
        string $endDate,
        string $externalId
    ): ?array
    {
        if (empty($this->apiToken) || empty($this->apiEndpoint) || $this->apiEndpoint === '/') {
            Log::warning('ZimflowService::sendReservationReminder called but service is not configured.');
            return ['status' => 'error', 'message' => 'Zimflow service (token/endpoint) not configured.'];
        }

        $payload = [
            'token' => $this->apiToken,
            'phone' => $phone,
            'name' => $name,
            'source' => $source,
            'start_date' => $startDate, // Expected format: Y-m-d H:i:s
            'end_date' => $endDate,     // Expected format: Y-m-d H:i:s
            'format' => 'Y-m-d H:i:s',
            'external_id' => $externalId,
        ];

        try {
            $response = Http::post("{$this->apiEndpoint}/api/reminders/reservation/makeReservation", $payload);

            if ($response->successful()) {
                return $response->json();
            }

            Log::error('Zimflow MakeReservation API error', [
                'status' => $response->status(),
                'body' => $response->body(),
                'payload' => array_merge($payload, ['token' => '***MASKED***'])
            ]);
            return ['status' => 'error', 'message' => 'API request failed.', 'details' => $response->json()];

        } catch (\Exception $e) {
            Log::error('Zimflow MakeReservation exception: ' . $e->getMessage(), ['payload' => array_merge($payload, ['token' => '***MASKED***'])]);
            return ['status' => 'error', 'message' => $e->getMessage()];
        }
    }

    /**
     * Send a generic message via Zimflow.
     *
     * @param string $recipient The recipient's identifier (e.g., phone number).
     * @param string $message The message content.
     * @return array|null The API response or null on failure.
     */
    public function sendGenericMessage(string $recipient, string $message): ?array
    {
        if (empty($this->apiToken) || empty($this->apiEndpoint) || $this->apiEndpoint === '/') {
            Log::warning('ZimflowService::sendGenericMessage called but service is not configured.');
            return ['status' => 'error', 'message' => 'Zimflow service (token/endpoint) not configured.'];
        }

        // Updated based on the provided Zimflow API documentation for sending a simple message.
        // Endpoint: {{api_endpoint}}api/wpbox/sendmessage
        // Method: POST
        // Auth: token in payload
        $actualGenericMessagePath = '/api/wpbox/sendmessage'; 
        $fullApiUrl = $this->apiEndpoint . $actualGenericMessagePath;

        $payload = [
            'token' => $this->apiToken,
            'phone' => $recipient, 
            'message' => $message,
            // Optional: 'buttons', 'header', 'footer' can be added here if needed for generic messages.
            // For a simple test notification, they are likely not required.
        ];

        try {
            $response = Http::post($fullApiUrl, $payload);

            if ($response->successful()) { // Log the successful response body for more insight
                Log::info('Zimflow SendGenericMessage successful API response', [
                    'url' => $fullApiUrl,
                    'response_body' => $response->json() // Or $response->body() if it's not always JSON
                ]);
                return $response->json();
            }
            Log::error('Zimflow SendGenericMessage API error', ['status' => $response->status(), 'body' => $response->body(), 'url' => $fullApiUrl, 'payload' => array_merge($payload, ['token' => '***MASKED***'])]);
            return ['status' => 'error', 'message' => 'API request failed.', 'details' => $response->json()];

        } catch (\Exception $e) {
            Log::error('Zimflow SendGenericMessage exception: ' . $e->getMessage());
            return ['status' => 'error', 'message' => $e->getMessage()];
        }
    }
}
