<?php

namespace Modules\StripeGateway\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Subscription;
use App\Models\SubscriptionPlan;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Modules\StripeGateway\Services\StripeService;
use App\Services\CreditService;
use App\Services\WalletService; // Added for wallet deposits
use Stripe\Checkout\Session as StripeCheckoutSession;

class StripeSubscriptionController extends Controller
{
    protected StripeService $stripeService;
    protected CreditService $creditService;
    protected WalletService $walletService; // Added

    public function __construct(StripeService $stripeService, CreditService $creditService, WalletService $walletService)
    {
        $this->stripeService = $stripeService;
        $this->creditService = $creditService;
        $this->walletService = $walletService; // Added
    }

    public function checkout(Request $request, SubscriptionPlan $plan)
    {
        if (setting('stripe_enabled', '0') != '1') {
            return redirect()->route('subscription.plans')->with('error', 'Stripe payments are currently disabled.');
        }

        /** @var User $user */
        $user = Auth::user();

        // Check if user already has an active subscription to this plan or any plan
        // (You might want more sophisticated logic here, e.g., allow upgrades/downgrades)
        $activeSubscription = $user->subscriptions()
                                ->whereIn('status', ['active', 'trialing'])
                                ->first();

        if ($activeSubscription) {
            // If they are trying to subscribe to the same plan they already have and it's active/trialing
            if ($activeSubscription->subscription_plan_id == $plan->id) {
                 return redirect()->route('subscription.plans')->with('info', 'You are already subscribed to this plan.');
            }
            // For now, prevent new subscriptions if one is active.
            // Later, you can implement upgrade/downgrade logic.
            return redirect()->route('subscription.plans')->with('error', 'You already have an active subscription. Please manage it from your profile.');
        }


        try {
            $checkoutSession = $this->stripeService->createCheckoutSession($user, $plan);
            return redirect($checkoutSession->url);
        } catch (\Exception $e) {
            Log::error("Stripe Checkout Error for user {$user->id}, plan {$plan->id}: " . $e->getMessage());
            return redirect()->route('subscription.plans')->with('error', 'Could not initiate Stripe checkout. ' . $e->getMessage());
        }
    }

    public function handleSuccess(Request $request)
    {
        $sessionId = $request->query('session_id');
        if (!$sessionId) {
            return redirect()->route('subscription.plans')->with('error', 'Invalid Stripe session.');
        }

        try {
            if (!setting('stripe_secret_key')) {
                 throw new \Exception('Stripe secret key is not configured.');
            }
            \Stripe\Stripe::setApiKey(setting('stripe_secret_key'));
            $session = StripeCheckoutSession::retrieve($sessionId, ['expand' => ['subscription', 'line_items.data.price.product']]);

            // Get the subscription ID from the session.
            // Based on the error, $session->subscription is a string (ID).
            $stripeSubscriptionIdFromSession = $session->subscription;

            if (!$stripeSubscriptionIdFromSession) {
                Log::error("Stripe success: Stripe Subscription ID not found in session.", ['session_id' => $sessionId, 'session_data' => $session->toArray()]);
                return redirect()->route('subscription.plans')->with('error', 'Subscription details could not be verified (missing subscription reference).');
            }

            // Retrieve the full Stripe Subscription object using the ID.
            // This ensures we have the object regardless of how `expand` behaved on the Session object.
            $stripeSubscriptionObject = \Stripe\Subscription::retrieve($stripeSubscriptionIdFromSession);

            $user = User::find($session->metadata->user_id);
            $plan = SubscriptionPlan::find($session->metadata->local_plan_id);

            if (!$user || !$plan) {
                Log::error("Stripe success: User or Plan not found from session metadata.", ['session_id' => $sessionId, 'metadata' => $session->metadata]);
                return redirect()->route('subscription.plans')->with('error', 'Subscription details could not be verified.');
            }

            // Check if a subscription record already exists for this Stripe subscription ID
            // This can happen if the webhook processed it first
            $existingSubscription = Subscription::where('gateway_subscription_id', $stripeSubscriptionObject->id)->first();

            if ($existingSubscription) {
                Log::info("Stripe success: Subscription already processed by webhook for Stripe sub ID: " . $stripeSubscriptionObject->id);
                 // Potentially update status if it was pending, though webhook should handle this.
                if ($existingSubscription->status === 'pending') {
                    $existingSubscription->status = $stripeSubscriptionObject->status; // e.g., 'active' or 'trialing'
                    $existingSubscription->starts_at = $stripeSubscriptionObject->current_period_start ? \Carbon\Carbon::createFromTimestamp($stripeSubscriptionObject->current_period_start) : now();
                    $existingSubscription->ends_at = $stripeSubscriptionObject->current_period_end ? \Carbon\Carbon::createFromTimestamp($stripeSubscriptionObject->current_period_end) : null;
                    if ($stripeSubscriptionObject->trial_end) {
                        $existingSubscription->trial_ends_at = \Carbon\Carbon::createFromTimestamp($stripeSubscriptionObject->trial_end);
                    }
                    $existingSubscription->save();
                }
            } else {
                // Create new local subscription record
                // Deactivate any other existing subscriptions for this user first
                $user->subscriptions()
                    ->whereIn('status', ['active', 'trialing'])
                    ->update([
                        'status' => 'cancelled',
                        'ends_at' => now(),
                        'cancelled_at' => now()
                    ]);

                $newSubscription = Subscription::create([
                    'user_id' => $user->id,
                    'subscription_plan_id' => $plan->id,
                    'payment_gateway' => 'stripegateway', // Consistent key
                    'gateway_subscription_id' => $stripeSubscriptionObject->id,
                    'status' => $stripeSubscriptionObject->status, // e.g., 'active' or 'trialing'
                    'price_at_purchase' => $plan->price,
                    'currency_at_purchase' => $plan->currency,
                    'starts_at' => $stripeSubscriptionObject->current_period_start ? \Carbon\Carbon::createFromTimestamp($stripeSubscriptionObject->current_period_start) : now(),
                    'ends_at' => $stripeSubscriptionObject->current_period_end ? \Carbon\Carbon::createFromTimestamp($stripeSubscriptionObject->current_period_end) : null,
                    'trial_ends_at' => $stripeSubscriptionObject->trial_end ? \Carbon\Carbon::createFromTimestamp($stripeSubscriptionObject->trial_end) : null,
                ]);

                // Award credits
                if (function_exists('setting') && setting('credits_system_enabled', '0') == '1' && $plan->credits_awarded_on_purchase > 0) {
                    $this->creditService->awardCredits($user, $plan->credits_awarded_on_purchase, 'award_subscription_purchase', "Credits for {$plan->name} subscription", $newSubscription);
                }

                // Assign target role
                if (!empty($plan->target_role) && class_exists(\Spatie\Permission\Models\Role::class) && \Spatie\Permission\Models\Role::where('name', $plan->target_role)->where('guard_name', 'web')->exists()) {
                    $user->syncRoles([$plan->target_role]);
                }
            }

            // Update user's Stripe customer ID if it's not set (though StripeService should do this)
            // Ensure we are assigning the customer ID, not the session ID.
            if ($user && $session->customer && ($user->stripe_customer_id !== $session->customer)) {
                $user->stripe_customer_id = $session->customer; // $session->customer is the Stripe Customer ID
                $user->save();
            }


            return redirect()->route('dashboard')->with('success', 'Subscription successful! Welcome to ' . $plan->name . '.');

        } catch (\Exception $e) {
            Log::error("Stripe Success Error: " . $e->getMessage(), ['session_id' => $sessionId]);
            return redirect()->route('subscription.plans')->with('error', 'There was an issue confirming your subscription: ' . $e->getMessage());
        }
    }

    public function handleCancel(Request $request)
    {
        return redirect()->route('subscription.plans')->with('info', 'Your subscription process was cancelled.');
    }
    
    public function redirectToCustomerPortal(Request $request)
    {
        /** @var User $user */
        $user = Auth::user();
        if (!$user->stripe_customer_id) {
            return redirect()->back()->with('error', 'Stripe customer profile not found.');
        }

        try {
            $portalSession = $this->stripeService->createBillingPortalSession($user);
            return redirect($portalSession->url);
        } catch (\Exception $e) {
            Log::error("Stripe Billing Portal Error for user {$user->id}: " . $e->getMessage());
            return redirect()->back()->with('error', 'Could not access the billing portal: ' . $e->getMessage());
        }
    }

    public function cancelActiveSubscription(Request $request)
    {
        /** @var User $user */
        $user = Auth::user();
    $activeSubscription = $user->subscriptions()
            ->where('payment_gateway', 'stripegateway') // Consistent with creation
            ->whereIn('status', ['active', 'trialing'])
            ->orderBy('created_at', 'desc')
            ->first();

        if (!$activeSubscription) {
            return redirect()->back()->with('info', 'No active Stripe subscription found to cancel.');
        }

        try {
            $cancelledAtGateway = $this->stripeService->cancelSubscriptionAtGateway($activeSubscription);
            if ($cancelledAtGateway) {
                // Webhook should ideally handle the final status update,
                // but we can mark it as 'cancelled' locally for immediate feedback.
                $activeSubscription->status = 'cancelled'; // Or 'pending_cancellation'
                $activeSubscription->cancelled_at = now();
                // ends_at will be updated by webhook when Stripe confirms cancellation at period end
                $activeSubscription->save();
                return redirect()->back()->with('success', 'Your subscription has been scheduled for cancellation at the end of the current billing period.');
            } else {
                return redirect()->back()->with('error', 'Failed to cancel subscription at Stripe. Please try again or contact support.');
            }
        } catch (\Exception $e) {
            Log::error("Error cancelling active Stripe subscription for user {$user->id}, sub ID {$activeSubscription->gateway_subscription_id}: " . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while trying to cancel your subscription: ' . $e->getMessage());
        }
    }

    // --- Wallet Deposit Methods ---

    public function initializeWalletDeposit(Request $request)
    {
        if (setting('stripe_enabled', '0') != '1' || setting('allow_wallet_deposits', '0') != '1') {
            return redirect()->route('user.wallet.deposit.form')->with('error', 'Stripe deposits are currently disabled.');
        }

        $request->validate(['amount' => 'required|numeric|min:1']); // Min deposit amount
        $amount = (float) $request->input('amount');
        /** @var User $user */
        $user = Auth::user();

        try {
            $checkoutSession = $this->stripeService->createWalletDepositCheckoutSession($user, $amount, strtolower(setting('currency_code', 'USD')));
            return redirect($checkoutSession->url);
        } catch (\Exception $e) {
            Log::error("Stripe Wallet Deposit Init Error for user {$user->id}, amount {$amount}: " . $e->getMessage());
            return redirect()->route('user.wallet.deposit.form')->with('error', 'Could not initiate Stripe deposit: ' . $e->getMessage());
        }
    }

    public function handleWalletDepositSuccess(Request $request)
    {
        $sessionId = $request->query('session_id');
        /** @var User $user */
        $user = Auth::user(); // Assuming user is logged in to complete this

        if (!$sessionId) {
            return redirect()->route('user.wallet.deposit.form')->with('error', 'Invalid Stripe session for deposit.');
        }

        try {
            $session = $this->stripeService->retrieveCheckoutSession($sessionId);

            if ($session && $session->payment_status == 'paid' && isset($session->metadata->type) && $session->metadata->type === 'wallet_deposit') {
                // Idempotency: Check if this payment_intent has already been processed
                $existingTransaction = \App\Models\WalletTransaction::where('gateway_transaction_id', $session->payment_intent)->first();
                if ($existingTransaction) {
                    Log::info("Stripe Wallet Deposit: Payment intent {$session->payment_intent} already processed.");
                    return redirect()->route('user.wallet.history')->with('success', 'Your deposit was successful.');
                }

                $amountDeposited = $session->amount_total / 100; // Stripe amount is in cents
                $currency = strtoupper($session->currency);

                $this->walletService->deposit($user, $amountDeposited, $currency, 'stripegateway', $session->payment_intent, "Wallet deposit via Stripe");
                return redirect()->route('user.wallet.history')->with('success', 'Successfully deposited ' . $currency . ' ' . number_format($amountDeposited, 2) . ' to your wallet.');
            }
            Log::error('Stripe Wallet Deposit: Payment not successful or session invalid/mismatched.', ['session_id' => $sessionId, 'payment_status' => $session->payment_status ?? 'N/A', 'metadata' => $session->metadata ?? []]);
            return redirect()->route('user.wallet.deposit.form')->with('error', 'Stripe payment verification failed for deposit.');
        } catch (\Exception $e) {
            Log::error("Stripe Wallet Deposit Success Error: " . $e->getMessage(), ['session_id' => $sessionId]);
            return redirect()->route('user.wallet.deposit.form')->with('error', 'There was an issue confirming your deposit: ' . $e->getMessage());
        }
    }

    public function handleWalletDepositCancel(Request $request)
    {
        return redirect()->route('user.wallet.deposit.form')->with('info', 'Your wallet deposit was cancelled.');
    }
}
