<?php

namespace Modules\PaystackGateway\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\SubscriptionPlan; // Assuming your SubscriptionPlan model
use App\Models\User;             // Assuming your User model


class PaystackService
{
    protected $publicKey;
        public $secretKey; // Make public or add a getter method
    protected $baseUrl = 'https://api.paystack.co';

    public function __construct()
    {
        if (!function_exists('setting')) {
            throw new \Exception("Settings helper function not found. Ensure it's globally available.");
        }
        $this->publicKey = setting('paystack_public_key');
        $this->secretKey = setting('paystack_secret_key');
        // Note: Paystack doesn't have a 'mode' in the API calls like PayPal SDK.
        // The live/test distinction is handled by using live or test keys.
    }

    protected function makeRequest(string $method, string $uri, array $data = [])
    {
        if (!$this->secretKey) {
            Log::error('Paystack secret key is not configured.');
            return null; // Or throw an exception
        }

        $response = Http::withToken($this->secretKey)
                        ->baseUrl($this->baseUrl)
                        ->{$method}($uri, $data);

        if (!$response->successful()) {
            Log::error("Paystack API Error: {$uri}", [
                'status' => $response->status(),
                'response' => $response->json() ?? $response->body()
            ]);
            // You might want to throw a custom exception here
            return null;
        }
        return $response->json();
    }

    public function initializeTransaction(int $amountKobo, string $email, string $reference, string $callbackUrl, string $currency = 'NGN', array $metadata = [])
    {
        $data = [
            'amount' => $amountKobo,
            'email' => $email,
            'reference' => $reference,
            'callback_url' => $callbackUrl,
            'currency' => strtoupper($currency), // Add currency here
            'metadata' => json_encode($metadata) // Paystack expects metadata as a JSON string
        ];
        // You can add more parameters like 'channels', 'currency' (NGN is default) if needed
        return $this->makeRequest('post', 'transaction/initialize', $data);
    }

    public function verifyTransaction(string $reference)
    {
        return $this->makeRequest('get', "transaction/verify/{$reference}");
    }

    public function createPlan(string $name, string $interval, int $amountKobo, string $currency, string $description = null)
    {
        // Paystack intervals: hourly, daily, weekly, monthly, biannually, annually
        // Your system intervals: day, week, month, year. Need mapping.
        $paystackIntervalMap = [
            'day' => 'daily',
            'week' => 'weekly',
            'month' => 'monthly',
            'year' => 'annually',
        ];
        $paystackInterval = $paystackIntervalMap[strtolower($interval)] ?? 'monthly';

        return $this->makeRequest('post', 'plan', [
            'name' => $name,
            'interval' => $paystackInterval,
            'amount' => $amountKobo, // Amount in Kobo
            'currency' => strtoupper($currency), // Add currency here
            'description' => $description,
        ]);
    }

    /**
     * Initialize a subscription transaction with Paystack.
     *
     * @param User $user
     * @param SubscriptionPlan $localPlan
     * @param int|null $localSubscriptionId The ID of your local pending subscription record to include in metadata.
     * @return array|null An array containing 'authorization_url' and 'reference', or null on failure.
     * @throws \Exception
     */
    public function initializeSubscriptionTransaction(User $user, SubscriptionPlan $localPlan, ?int $localSubscriptionId = null): ?array
    {
        if (!$this->secretKey) {
            Log::error('PaystackService: Secret key is not configured.');
            throw new \Exception('Paystack payment gateway is not configured properly.');
        }

        $paystackPlanCode = $localPlan->paystack_plan_code;

        if (empty($paystackPlanCode)) {
            // Plan doesn't exist on Paystack yet, create it.
            $amountKobo = (int) round($localPlan->price * 100); // Ensure price is converted to Kobo
            $currencyForPaystackPlan = strtoupper($localPlan->currency ?? setting('currency_code', 'USD')); // Use plan's currency
            $planName = $localPlan->name . ' - ' . $localPlan->interval . ($localPlan->interval_count > 1 ? ' ' . $localPlan->interval_count : '');
            $paystackPlan = $this->createPlan($planName, $localPlan->interval, $amountKobo, $currencyForPaystackPlan, $localPlan->description);

            if ($paystackPlan && isset($paystackPlan['status']) && $paystackPlan['status'] === true && isset($paystackPlan['data']['plan_code'])) {
                $paystackPlanCode = $paystackPlan['data']['plan_code'];
                $localPlan->paystack_plan_code = $paystackPlanCode;
                $localPlan->save(); // Save the Paystack plan code to your local plan
                Log::info("PaystackService: Created Paystack plan '{$paystackPlanCode}' for local plan ID {$localPlan->id}");
            } else {
                Log::error("PaystackService: Failed to create Paystack plan for local plan ID {$localPlan->id}", ['response' => $paystackPlan]);
                throw new \Exception('Failed to create payment plan on Paystack.');
            }
        }

        // Now initialize the transaction with the Paystack plan code
        $reference = 'SUB__' . $user->id . '__' . $localPlan->id . '__' . time() . '__' . uniqid();
        $callbackUrl = route('subscription.paystack.callback'); // Ensure this route exists in your PaystackGateway module

        $data = [
            'email' => $user->email,
            // Amount for the first transaction. If the plan has a price, Paystack uses the plan's price.
            // This amount is an optional initial charge or if the plan is free and you want to charge an initial fee.
            // For simplicity, we'll pass the plan's price. Paystack will use the plan's defined amount.
            'amount' => (int) round($localPlan->price * 100),
            'plan' => $paystackPlanCode, // Key for subscription
            'reference' => $reference,
            'callback_url' => $callbackUrl,
            'metadata' => json_encode([
                'user_id' => $user->id,
                'local_plan_id' => $localPlan->id,
                'subscription_id' => $localSubscriptionId, // Pass your local subscription ID
                'custom_fields' => [ // Paystack's recommended structure for metadata
                    ['display_name' => "User Name", "variable_name" => "user_name", "value" => $user->name],
                    ['display_name' => "Plan Name", "variable_name" => "plan_name", "value" => $localPlan->name]
                ]
            ])
        ];

        $response = $this->makeRequest('post', 'transaction/initialize', $data);

        if ($response && isset($response['status']) && $response['status'] === true && isset($response['data']['authorization_url'], $response['data']['reference'])) {
            return [
                'authorization_url' => $response['data']['authorization_url'],
                'reference' => $response['data']['reference'] // This is the reference Paystack will use
            ];
        }
        return null;
    }
}