<?php

namespace Modules\PaynowGateway\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Subscription;
use App\Models\SubscriptionPlan;
use App\Models\User;
use App\Services\CreditService;
use App\Services\WalletService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Modules\PaynowGateway\Services\PaynowService;
use Spatie\Permission\Models\Role;

class PaynowSubscriptionController extends Controller
{
    protected PaynowService $paynowService;
    protected CreditService $creditService;
    protected WalletService $walletService;

    public function __construct(
        PaynowService $paynowService,
        CreditService $creditService,
        WalletService $walletService
    ) {
        $this->paynowService = $paynowService;
        $this->creditService = $creditService;
        $this->walletService = $walletService;
    }

    public function initializeSubscriptionPayment(Request $request, SubscriptionPlan $plan)
    {
        if (setting('paynowgateway_enabled', '0') != '1') {
            return redirect()->route('subscription.plans')->with('error', 'Paynow payments are currently disabled.');
        }

        /** @var User $user */
        $user = Auth::user();
        $subscriptionReference = 'SUB-' . Str::uuid()->toString(); // Unique reference for this payment attempt

        try {
            // Create a pending local subscription record
            $pendingSubscription = Subscription::create([
                'user_id' => $user->id,
                'subscription_plan_id' => $plan->id,
                'payment_gateway' => 'paynow_gateway',
                'gateway_transaction_id' => $subscriptionReference, // Store our unique reference
                'status' => 'pending_payment',
                'price_at_purchase' => $plan->price,
                'currency_at_purchase' => $plan->currency, // Use plan's currency
                // gateway_poll_url will be set after successful initiation
            ]);

            $paymentDetails = $this->paynowService->initiateSubscriptionPayment(
                $user,
                $plan,
                $pendingSubscription->id,
                $subscriptionReference
            );

            if ($paymentDetails && $paymentDetails['status'] === 'success' && isset($paymentDetails['redirect_url'])) {
                // Update local subscription with Paynow's poll_url and potentially their reference
                $pendingSubscription->gateway_poll_url = $paymentDetails['poll_url'];
                // $pendingSubscription->gateway_transaction_id = $paymentDetails['paynow_reference']; // Or keep $subscriptionReference
                $pendingSubscription->save();
                return redirect()->away($paymentDetails['redirect_url']);
            }

            // If initiation failed at the service level
            $pendingSubscription->update(['status' => 'failed', 'notes' => 'Paynow initiation failed.']);
            Log::error('Paynow initialize subscription payment failed at controller.', [
                'service_response' => $paymentDetails, 'user_id' => $user->id, 'plan_id' => $plan->id
            ]);
            return redirect()->route('subscription.plans')->with('error', 'Could not initiate Paynow payment. Please try again.');

        } catch (\Exception $e) {
            if (isset($pendingSubscription) && $pendingSubscription->exists) {
                $pendingSubscription->update(['status' => 'failed', 'notes' => 'Error during Paynow initiation: ' . $e->getMessage()]);
            }
            Log::error("Paynow Initialize Subscription Payment Error for user {$user->id}, plan {$plan->id}: " . $e->getMessage());
            return redirect()->route('subscription.plans')->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }

    public function handleSubscriptionCallback(Request $request)
    {
        /** @var User $user */
        $user = Auth::user();
        $localSubscriptionId = $request->query('subscription_id');
        $transactionReference = $request->query('ref'); // This is our $subscriptionReference

        Log::info('Paynow Subscription Callback Received:', $request->all());

        if (!$localSubscriptionId || !$transactionReference) {
            Log::error('Paynow Subscription Callback: Missing subscription_id or ref in callback.', $request->all());
            return redirect()->route('subscription.plans')->with('error', 'Invalid callback data. Please contact support.');
        }

        $subscription = Subscription::where('id', $localSubscriptionId)
                                ->where('gateway_transaction_id', $transactionReference)
                                ->where('status', 'pending_payment')
                                ->first();

        if (!$subscription) {
            Log::warning('Paynow Subscription Callback: Local subscription not found, not pending, or reference mismatch.', [
                'local_subscription_id' => $localSubscriptionId, 'transaction_reference' => $transactionReference, 'user_id' => $user->id,
            ]);
            // Check if it was already processed
            $alreadyProcessedSub = Subscription::find($localSubscriptionId);
            if ($alreadyProcessedSub && $alreadyProcessedSub->status !== 'pending_payment') {
                 return redirect()->route('dashboard')->with('info', 'This subscription payment has already been processed.');
            }
            return redirect()->route('subscription.plans')->with('error', 'Subscription record issue. Please contact support.');
        }

        if (empty($subscription->gateway_poll_url)) {
            Log::error('Paynow Subscription Callback: Poll URL missing for subscription.', ['sub_id' => $subscription->id]);
            $subscription->update(['status' => 'failed', 'notes' => 'Callback received but Poll URL was missing.']);
            return redirect()->route('subscription.plans')->with('error', 'Cannot verify payment: Poll URL missing. Contact support.');
        }

        try {
            $verificationResult = $this->paynowService->verifyPaymentStatus($subscription->gateway_poll_url);
            Log::info('Paynow Subscription Callback - Verification Result:', $verificationResult);

            if ($verificationResult && $verificationResult['paid'] === true) {
                // Cancel other active/trialing subscriptions for the user
                $user->subscriptions()->where('id', '!=', $subscription->id)->whereIn('status', ['active', 'trialing'])->update(['status' => 'cancelled', 'ends_at' => now(), 'cancelled_at' => now()]);

                $plan = $subscription->plan;
                $subscription->status = $plan->trial_period_days > 0 ? 'trialing' : 'active';
                $subscription->starts_at = now();
                $subscription->trial_ends_at = $plan->trial_period_days > 0 ? now()->addDays($plan->trial_period_days) : null;
                $subscription->ends_at = now()->add($plan->interval, $plan->interval_count);
                $subscription->gateway_transaction_id = $verificationResult['paynow_reference'] ?? $subscription->gateway_transaction_id; // Update to Paynow's ref if available
                $subscription->notes = 'Payment successful via Paynow. Status: ' . $verificationResult['status'];
                $subscription->save();

                if (function_exists('setting') && setting('credits_system_enabled', '0') == '1' && $plan->credits_awarded_on_purchase > 0) {
                    $this->creditService->awardCredits($user, $plan->credits_awarded_on_purchase, 'award_subscription_purchase', "Credits for {$plan->name} subscription", $subscription);
                }
                if (!empty($plan->target_role) && class_exists(Role::class) && Role::where('name', $plan->target_role)->where('guard_name', 'web')->exists()) {
                    $user->syncRoles([$plan->target_role]);
                }
                return redirect()->route('dashboard')->with('success', 'Subscription successfully activated via Paynow!');
            } else {
                // Payment not confirmed as paid by Paynow
                $subscription->status = 'failed';
                $subscription->notes = 'Paynow payment verification failed or payment not completed. Status: ' . ($verificationResult['status'] ?? 'unknown') . '. Message: ' . ($verificationResult['message'] ?? '');
                $subscription->save();

                Log::error('Paynow subscription verification indicated failure or pending.', [
                    'verification_data' => $verificationResult, 'request_data' => $request->all(),
                    'subscription_id' => $subscription->id
                ]);
                return redirect()->route('subscription.plans')->with('error', $verificationResult['message'] ?? 'Paynow payment verification failed or payment not completed.');
            }

        } catch (\Exception $e) {
            Log::error("Paynow Subscription Callback Exception: " . $e->getMessage(), [
                'request_data' => $request->all(),
                'subscription_id' => $subscription->id ?? null, // Use null coalescing as $subscription might not be set if first() fails
            ]);
            if (isset($subscription) && $subscription->exists) { // Check if $subscription was successfully fetched and exists
                $subscription->status = 'failed';
                $subscription->notes = 'Error during Paynow callback processing: ' . $e->getMessage();
                $subscription->save();
            }
            return redirect()->route('subscription.plans')->with('error', 'An error occurred during payment verification.');
        }
    }

    public function handleSubscriptionCancel(Request $request)
    {
        // Optionally, find and update the pending subscription to 'cancelled' if a reference is passed back
        $transactionReference = $request->query('ref');
        if ($transactionReference) {
            $subscription = Subscription::where('gateway_transaction_id', $transactionReference)
                                        ->where('status', 'pending_payment')
                                        ->first();
            if ($subscription) {
                $subscription->update(['status' => 'cancelled', 'notes' => 'User cancelled payment process on Paynow.']);
            }
        }
        return redirect()->route('subscription.plans')->with('info', 'Paynow subscription process was cancelled.');
    }

    // --- Wallet Deposit Methods ---

    public function initializeWalletDeposit(Request $request)
    {
        if (setting('paynowgateway_enabled', '0') != '1' || setting('allow_wallet_deposits', '0') != '1') {
            return redirect()->route('user.wallet.deposit.form')->with('error', 'Paynow deposits are currently disabled.');
        }

        $minDeposit = setting('wallet_min_deposit_amount', 1);
        $validated = $request->validate(['amount' => "required|numeric|min:{$minDeposit}"]);
        $amount = (float) $validated['amount'];
        /** @var User $user */
        $user = Auth::user();
        $currency = strtoupper(setting('wallet_default_currency', 'USD'));
        $transactionReference = 'WLT-' . Str::uuid()->toString();

        try {
            $paymentDetails = $this->paynowService->initiateOneTimePayment(
                $user,
                $amount,
                $currency,
                "Wallet Deposit for user {$user->email}",
                $transactionReference
            );

            if ($paymentDetails && $paymentDetails['status'] === 'success' && isset($paymentDetails['redirect_url'])) {
                // Store necessary details in session for callback verification
                session([
                    'paynow_wallet_reference' => $transactionReference,
                    'paynow_wallet_poll_url' => $paymentDetails['poll_url'],
                    'paynow_wallet_amount' => $amount,
                    'paynow_wallet_currency' => $currency,
                ]);
                return redirect()->away($paymentDetails['redirect_url']);
            }
            Log::error('Paynow initialize wallet deposit failed at controller.', [
                'service_response' => $paymentDetails, 'user_id' => $user->id, 'amount' => $amount
            ]);
            return redirect()->route('user.wallet.deposit.form')->with('error', 'Could not initiate Paynow deposit.');
        } catch (\Exception $e) {
            Log::error("Paynow Initialize Wallet Deposit Error for user {$user->id}, amount {$amount}: " . $e->getMessage());
            return redirect()->route('user.wallet.deposit.form')->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }

    public function handleWalletDepositCallback(Request $request)
    {
        /** @var User $user */
        $user = Auth::user();
        $transactionReference = $request->query('ref'); // This is our $transactionReference

        // Retrieve details from session
        $sessionReference = session('paynow_wallet_reference');
        $pollUrl = session('paynow_wallet_poll_url');
        $amount = session('paynow_wallet_amount');
        $currency = session('paynow_wallet_currency');

        Log::info('Paynow Wallet Deposit Callback Received:', $request->all());

        if (!$transactionReference || $transactionReference !== $sessionReference || !$pollUrl || !$amount) {
            Log::error('Paynow Wallet Callback: Invalid or missing session data or reference mismatch.', [
                'request_ref' => $transactionReference, 'session_ref' => $sessionReference,
                'poll_url_present' => !empty($pollUrl), 'user_id' => $user->id,
            ]);
            $this->clearWalletSessionData();
            return redirect()->route('user.wallet.deposit.form')->with('error', 'Invalid deposit session or callback. Please try again.');
        }

        try {
            $verificationResult = $this->paynowService->verifyPaymentStatus($pollUrl);
            Log::info('Paynow Wallet Deposit Callback - Verification Result:', $verificationResult);

            if ($verificationResult && $verificationResult['paid'] === true) {
                // Optional: Verify amount if Paynow returns it in verification
                $paidAmount = (float) ($verificationResult['amount'] ?? $amount); // Use session amount if not in verification
                if ($paidAmount < $amount) { // Check if paid amount is less than expected
                    Log::warning('Paynow Wallet Deposit: Amount mismatch.', [
                        'expected' => $amount, 'paid' => $paidAmount, 'user_id' => $user->id, 'ref' => $transactionReference
                    ]);
                    // Decide how to handle amount mismatch, e.g., log and proceed, or error out
                    // For now, we'll proceed with the session amount if Paynow confirms 'paid'
                }

                $this->walletService->deposit(
                    $user,
                    $amount, // Use the originally intended amount
                    $currency,
                    'paynow_gateway',
                    $verificationResult['paynow_reference'] ?? $transactionReference, // Use Paynow's ref if available
                    "Wallet deposit via Paynow. Ref: {$transactionReference}"
                );

                $this->clearWalletSessionData();
                return redirect()->route('user.wallet.history')->with('success', 'Wallet deposit successful via Paynow!');
            } else {
                Log::error('Paynow wallet deposit verification indicated failure or pending.', [
                    'verification_data' => $verificationResult, 'user_id' => $user->id, 'ref' => $transactionReference
                ]);
                $this->clearWalletSessionData();
                return redirect()->route('user.wallet.deposit.form')->with('error', $verificationResult['message'] ?? 'Paynow deposit verification failed or payment not completed.');
            }
        } catch (\Exception $e) {
            Log::error("Paynow Wallet Deposit Callback Exception: " . $e->getMessage(), [
                'request_data' => $request->all(), 'user_id' => $user->id, 'ref' => $transactionReference
            ]);
            $this->clearWalletSessionData();
            return redirect()->route('user.wallet.deposit.form')->with('error', 'An error occurred during deposit verification.');
        }
    }

    private function clearWalletSessionData()
    {
        session()->forget([
            'paynow_wallet_reference',
            'paynow_wallet_poll_url',
            'paynow_wallet_amount',
            'paynow_wallet_currency'
        ]);
    }

    public function handleWalletDepositCancel(Request $request)
    {
        $this->clearWalletSessionData();
        return redirect()->route('user.wallet.deposit.form')->with('info', 'Paynow wallet deposit was cancelled.');
    }

    // --- Credit Purchase Methods ---

    public function initializeCreditPurchase(Request $request)
    {
        if (setting('paynowgateway_enabled', '0') != '1' || setting('credits_system_enabled', '0') != '1') {
            return redirect()->route('user.credits.topup.form')->with('error', 'Paynow credit purchases are currently disabled.');
        }
        // Optional: Check a specific setting like 'paynowgateway_allow_credit_purchases'

        $minCredits = setting('credits_min_purchase_amount', 1);
        $validated = $request->validate(['credits' => "required|integer|min:{$minCredits}"]);
        $creditsToPurchase = (int) $validated['credits'];

        /** @var User $user */
        $user = Auth::user();

        // Determine the monetary amount for these credits
        // This is crucial. You need a setting for "price_per_credit" or credit packages.
        $pricePerCredit = (float) setting('price_per_credit', 0.0); // Default to 0, which means it must be set
        if ($pricePerCredit <= 0) {
            Log::error("PaynowGateway: Price per credit is not configured or is zero. Cannot process credit purchase.");
            return redirect()->route('user.credits.topup.form')->with('error', 'Credit pricing is not configured. Please contact support.');
        }
        $amountToPay = $creditsToPurchase * $pricePerCredit;
        $currency = strtoupper(setting('currency_code', 'USD')); // Use global currency or a specific credit purchase currency
        $transactionReference = 'CRD-' . Str::uuid()->toString();

        try {
            $paymentDetails = $this->paynowService->initiateOneTimePayment(
                $user,
                $amountToPay,
                $currency,
                "Purchase of {$creditsToPurchase} credits",
                $transactionReference
            );

            if ($paymentDetails && $paymentDetails['status'] === 'success' && isset($paymentDetails['redirect_url'])) {
                session([
                    'paynow_credit_purchase_reference' => $transactionReference,
                    'paynow_credit_purchase_poll_url' => $paymentDetails['poll_url'],
                    'paynow_credit_purchase_credits' => $creditsToPurchase, // Store credits being purchased
                    'paynow_credit_purchase_amount_paid' => $amountToPay,
                    'paynow_credit_purchase_currency' => $currency,
                ]);
                return redirect()->away($paymentDetails['redirect_url']);
            }
            Log::error('Paynow initialize credit purchase failed at controller.', [
                'service_response' => $paymentDetails, 'user_id' => $user->id, 'credits' => $creditsToPurchase, 'amount' => $amountToPay
            ]);
            return redirect()->route('user.credits.topup.form')->with('error', 'Could not initiate Paynow payment for credits.');
        } catch (\Exception $e) {
            Log::error("Paynow Initialize Credit Purchase Error for user {$user->id}, credits {$creditsToPurchase}: " . $e->getMessage());
            return redirect()->route('user.credits.topup.form')->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }

    public function handleCreditPurchaseCallback(Request $request)
    {
        /** @var User $user */
        $user = Auth::user();
        $transactionReference = $request->query('ref');

        $sessionReference = session('paynow_credit_purchase_reference');
        $pollUrl = session('paynow_credit_purchase_poll_url');
        $creditsPurchased = session('paynow_credit_purchase_credits');

        Log::info('Paynow Credit Purchase Callback Received:', $request->all());

        if (!$transactionReference || $transactionReference !== $sessionReference || !$pollUrl || !$creditsPurchased) {
            Log::error('Paynow Credit Purchase Callback: Invalid or missing session data or reference mismatch.', ['request_ref' => $transactionReference, 'session_ref' => $sessionReference, 'user_id' => $user->id]);
            session()->forget(['paynow_credit_purchase_reference', 'paynow_credit_purchase_poll_url', 'paynow_credit_purchase_credits', 'paynow_credit_purchase_amount_paid', 'paynow_credit_purchase_currency']);
            return redirect()->route('user.credits.topup.form')->with('error', 'Invalid credit purchase session or callback.');
        }

        try {
            $verificationResult = $this->paynowService->verifyPaymentStatus($pollUrl);
            Log::info('Paynow Credit Purchase Callback - Verification Result:', $verificationResult);

            if ($verificationResult && $verificationResult['paid'] === true) {
                $this->creditService->awardCredits($user, $creditsPurchased, 'credit_purchase', "Purchased {$creditsPurchased} credits via Paynow. Ref: " . ($verificationResult['paynow_reference'] ?? $transactionReference));
                session()->forget(['paynow_credit_purchase_reference', 'paynow_credit_purchase_poll_url', 'paynow_credit_purchase_credits', 'paynow_credit_purchase_amount_paid', 'paynow_credit_purchase_currency']);
                return redirect()->route('user.credits.history')->with('success', "Successfully purchased {$creditsPurchased} credits via Paynow!");
            }
            Log::error('Paynow credit purchase verification failed or payment not completed.', ['verification_data' => $verificationResult, 'user_id' => $user->id, 'ref' => $transactionReference]);
            session()->forget(['paynow_credit_purchase_reference', 'paynow_credit_purchase_poll_url', 'paynow_credit_purchase_credits', 'paynow_credit_purchase_amount_paid', 'paynow_credit_purchase_currency']);
            return redirect()->route('user.credits.topup.form')->with('error', $verificationResult['message'] ?? 'Paynow payment verification failed for credit purchase.');
        } catch (\Exception $e) {
            Log::error("Paynow Credit Purchase Callback Exception: " . $e->getMessage(), ['request_data' => $request->all(), 'user_id' => $user->id, 'ref' => $transactionReference]);
            session()->forget(['paynow_credit_purchase_reference', 'paynow_credit_purchase_poll_url', 'paynow_credit_purchase_credits', 'paynow_credit_purchase_amount_paid', 'paynow_credit_purchase_currency']);
            return redirect()->route('user.credits.topup.form')->with('error', 'An error occurred during credit purchase verification.');
        }
    }
}
