<?php

namespace Modules\LandingPage\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Gate;
use Modules\LandingPage\Entities\LpPage; // Assuming this is your model
use Modules\LandingPage\Policies\LpPagePolicy; // Assuming this is your policy
// use Illuminate\Database\Eloquent\Factory; // Uncomment if you use factories in this module
use Illuminate\Support\Facades\Blade;

class LandingPageServiceProvider extends ServiceProvider
{
    /**
     * @var string $moduleName
     */
    protected string $moduleName = 'LandingPage';

    /**
     * @var string $moduleNameLower
     */
    protected string $moduleNameLower = 'landingpage';

    /**
     * Boot the application events.
     *
     * @return void
     */
    public function boot(): void
    {
        $this->registerTranslations();
        $this->registerConfig();
        $this->registerViews();
        $this->loadMigrationsFrom(module_path($this->moduleName, 'Database/Migrations'));
        // Route loading for web and api is now handled by Modules\LandingPage\Providers\RouteServiceProvider::class
        // Admin routes are loaded by the main application's RouteServiceProvider.
        $this->publishAssets();
        $this->registerBladeComponents();
        $this->registerPolicies();
    }

    /**
     * Register the service provider.
     *
     * @return void
     */
    public function register(): void
    {
        // Register the module's RouteServiceProvider to handle its web and api routes.
        $this->app->register(\Modules\LandingPage\Providers\RouteServiceProvider::class);
    }

    /**
     * Register config.
     *
     * @return void
     */
    protected function registerConfig(): void
    {
        $this->publishes([
            module_path($this->moduleName, 'config/config.php') => config_path($this->moduleNameLower . '.php'),
        ], 'config');
        $this->mergeConfigFrom(
            module_path($this->moduleName, 'config/config.php'), $this->moduleNameLower
        );
    }

    /**
     * Register views.
     *
     * @return void
     */
    public function registerViews(): void
    {
        $viewPath = resource_path('views/modules/' . $this->moduleNameLower);
        $sourcePath = module_path($this->moduleName, 'resources/views');

        $this->publishes([
            $sourcePath => $viewPath
        ], ['views', $this->moduleNameLower . '-module-views']);

        $this->loadViewsFrom(array_merge($this->getPublishableViewPaths(), [$sourcePath]), $this->moduleNameLower);
    }

    /**
     * Register translations.
     *
     * @return void
     */
    public function registerTranslations(): void
    {
        $langPath = resource_path('lang/modules/' . $this->moduleNameLower);

        if (is_dir($langPath)) {
            $this->loadTranslationsFrom($langPath, $this->moduleNameLower);
            $this->loadJsonTranslationsFrom($langPath);
        } else {
            $this->loadTranslationsFrom(module_path($this->moduleName, 'lang'), $this->moduleNameLower);
            $this->loadJsonTranslationsFrom(module_path($this->moduleName, 'lang'));
        }
    }

    private function getPublishableViewPaths(): array
    {
        $paths = [];
        foreach (\Config::get('view.paths') as $path) {
            if (is_dir($path . '/modules/' . $this->moduleNameLower)) {
                $paths[] = $path . '/modules/' . $this->moduleNameLower;
            }
        }
        return $paths;
    }

    protected function publishAssets(): void
    {
        $this->publishes([
            module_path($this->moduleName, 'public') => public_path('modules/'. $this->moduleNameLower),
        ], ['public', $this->moduleNameLower . '-module-assets']);
    }

    protected function registerBladeComponents(): void
    {
        // Example: Blade::component('landingpage::components.alert', 'lp-alert');
    }

    public function registerPolicies()
    {
        Gate::policy(LpPage::class, LpPagePolicy::class);
    }

    public function provides(): array
    {
        return [];
    }
}
