<?php

namespace Modules\LandingPage\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Modules\LandingPage\Entities\LpPage; // Make sure to import LpPage

class MenuItem extends Model
{
    use HasFactory;

    protected $table = 'lp_menu_items'; // Explicitly define table name
    protected $fillable = [
        'menu_id',
        'title',
        'type', // 'page', 'url', 'route'
        'url',
        'route_name', // Changed from 'route' for consistency
        'parameters',
        'lp_page_id', 
        'target', 'icon_class', 'css_class', 'parent_id', 'sort_order',
        'is_active', // Add this
    ];

    protected $casts = [
        'parameters' => 'array',
        'is_active' => 'boolean', // Add this
    ];

    protected static function newFactory()
    {
        // If you create a factory, you can uncomment and point to it:
        // return \Modules\LandingPage\Database\factories\MenuItemFactory::new();
        return null;
    }

    /**
     * Get the menu that this item belongs to.
     */
    public function menu()
    {
        return $this->belongsTo(Menu::class, 'menu_id');
    }

    /**
     * Get the landing page that this menu item links to (if any).
     */
    public function page()
    {
        return $this->belongsTo(LpPage::class, 'lp_page_id');
    }

    /**
     * Get the parent menu item (if this is a sub-item).
     */
    public function parent()
    {
        return $this->belongsTo(MenuItem::class, 'parent_id');
    }

    /**
     * Get the children of this menu item (for sub-menus), ordered by 'order'.
     */
    public function children()
    {
        return $this->hasMany(MenuItem::class, 'parent_id')->orderBy('sort_order'); // Changed to 'sort_order'
    }

    /**
     * Recursive relationship to load all descendants, ordered by 'order'.
     */
    public function childrenRecursive()
    {
        return $this->children()->with('childrenRecursive');
    }

    /**
     * Accessor to get the actual URL for the menu item.
     * Used in frontend rendering.
     */
    public function getLinkAttribute(): string
    {
        switch ($this->type) {
            case 'page':
                return $this->page ? route('landingpage.page.show', $this->page->slug) : '#';
            case 'url':
                return $this->url ?: '#';
            case 'route':
                return $this->route_name ? route($this->route_name, $this->parameters ?: []) : '#';
            default:
                return '#';
        }
    }
}
