<?php

namespace Modules\DigitalVocano\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Str;

class Project extends Model
{
    use HasFactory;

    protected $table = 'dv_projects'; // Table name with prefix

    protected $fillable = [
        'name',
        'slug',
        'short_description',
        'full_description',
        'featured_image', // Will store the URL/path to the image
        'store_url',
        'demo_url',
        // 'documentation_url', // External documentation link - REMOVED as per your request
        'internal_documentation_content', // For documentation managed within the module
        'is_published',
        'sort_order',
    ];

    protected $casts = [
        'is_published' => 'boolean',
        'features' => 'array', // If you add a features JSON column later
    ];

    /**
     * The "booted" method of the model.
     *
     * @return void
     */
    protected static function booted(): void
    {
        static::creating(function ($project) {
            if (empty($project->slug)) {
                $project->slug = Str::slug($project->name);
            }
            // Ensure slug uniqueness on creation
            $originalSlug = $project->slug;
            $count = 1;
            while (static::where('slug', $project->slug)->exists()) {
                $project->slug = $originalSlug . '-' . $count++;
            }
        });

        static::updating(function ($project) {
            // If the name is changed and slug is not explicitly set, regenerate slug
            if ($project->isDirty('name') && !$project->isDirty('slug')) {
                $project->slug = Str::slug($project->name);
            }
            // Ensure slug uniqueness on update if it has changed
            if ($project->isDirty('slug')) {
                $originalSlug = $project->slug;
                $count = 1;
                while (static::where('slug', $project->slug)->where('id', '!=', $project->id)->exists()) {
                    $project->slug = $originalSlug . '-' . $count++;
                }
            }
        });
    }

    /**
     * Get the URL for the featured image.
     *
     * @return string|null
     */
    public function getFeaturedImageUrlAttribute(): ?string
    {
        if ($this->featured_image) {
            // Assuming featured_image stores a path relative to the storage/app/public directory
            // and you have run `php artisan storage:link`
            // If it's already a full URL, you might not need Storage::url()
            if (Str::startsWith($this->featured_image, ['http://', 'https://'])) {
                return $this->featured_image;
            }
            // Ensure the path doesn't start with 'public/' if Storage::url is used
            $imagePath = Str::startsWith($this->featured_image, 'public/')
                ? Str::after($this->featured_image, 'public/')
                : $this->featured_image;

            return \Illuminate\Support\Facades\Storage::disk('public')->url($imagePath);
        }
        return null; // Or a path to a default placeholder image
    }


    /**
     * Define a new factory
     *
     * @return \Modules\DigitalVocano\Database\factories\ProjectFactory
     */
    protected static function newFactory()
    {
        // You'll need to create this factory if you plan to use seeders or model factories.
        // Run: php artisan module:make-factory ProjectFactory DigitalVocano
        return \Modules\DigitalVocano\Database\factories\ProjectFactory::new();
    }

    // Add any other relationships or helper methods here
    // For example, if you create a separate DocumentationPage model:
    // public function documentationPages()
    // {
    //     return $this->hasMany(DocumentationPage::class)->orderBy('order_column');
    // }
}
