<?php

namespace Modules\BusinessDirectory\Policies;

use App\Models\User;
use Modules\BusinessDirectory\Entities\Tender;
use Modules\BusinessDirectory\Entities\Company;
use Illuminate\Auth\Access\HandlesAuthorization;

class TenderPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any models.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function viewAny(User $user)
    {
        // Anyone can view the list of tenders (public index page)
        return true;
    }

    /**
     * Determine whether the user can view the model.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Tender  $tender
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function view(?User $user, Tender $tender)
    {
        // Allow viewing if the tender is open, or if the user owns it
        if ($tender->status === 'open') {
            return true;
        }
        return $user && $user->id === $tender->user_id;
    }

    /**
     * Determine whether the user can create models.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function create(User $user)
    {
        // Allow users to create tenders if they have at least one approved company.
        // The controller already checks if $userCompanies is empty.
        // You might add more conditions here, e.g., subscription status.
        return $user->companies()->where('status', 'approved')->exists();
    }

    /**
     * Determine whether the user can create a tender for a specific company.
     * This is used in the store method.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Company  $company
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function createForCompany(User $user, Company $company)
    {
        return $user->id === $company->user_id && $company->status === 'approved';
    }

    /**
     * Determine whether the user can update the model.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Tender  $tender
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function update(User $user, Tender $tender)
    {
        return $user->id === $tender->user_id; // Only the user who created the tender can update it
    }

    /**
     * Determine whether the user can delete the model.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Tender  $tender
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function delete(User $user, Tender $tender)
    {
        return $user->id === $tender->user_id; // Only the user who created the tender can delete it
    }

    /**
     * Determine whether the user can manage bids for the tender.
     */
    public function manageBids(User $user, Tender $tender): bool
    {
        return $user->id === $tender->user_id;
    }
}