<?php

namespace Modules\BusinessDirectory\Policies;

use App\Models\User; // Assuming your User model path is App\Models\User
use Illuminate\Auth\Access\HandlesAuthorization;
use Modules\BusinessDirectory\Entities\Review;
use Modules\BusinessDirectory\Entities\Company; // Needed to check company

class ReviewPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view the review.
     * Public reviews are viewable by anyone. This is for unapproved reviews.
     */
    public function view(?User $user, Review $review): bool
    {
        // Approved reviews are publicly viewable
        if ($review->is_approved) {
            return true;
        }

        // Allow the user who wrote the review to view it
        if ($user && $review->user_id === $user->id) {
            return true;
        }

        // Allow the owner of the company being reviewed to view it
        if ($user && $review->company && $review->company->user_id === $user->id) {
            return true;
        }

        // Allow administrators to view any review
        if ($user && (isset($user->is_admin) && $user->is_admin)) { // Assuming User model has an is_admin flag
            return true;
        }

        return false; // Default deny
    }

    /**
     * Determine whether the user can create reviews for a company.
     * This policy method receives the Review class and the Company instance.
     */
    public function create(User $user, Company $company): bool
    {
        // User must be authenticated (handled by middleware)
        // User cannot review their own company
        if ($company->user_id === $user->id) {
            return false;
        }

        // User cannot review the same company multiple times (checked in controller)

        // TODO: Add eligibility check (e.g., must have interacted with the company)

        return true; // Basic check: authenticated user can review other companies
    }

    /**
     * Determine whether the user can update the review.
     */
    public function update(User $user, Review $review): bool
    {
        // User must be the author of the review
        return $review->user_id === $user->id;
    }

    /**
     * Determine whether the user can delete the review.
     */
    public function delete(User $user, Review $review): bool
    {
        // User must be the author of the review OR the owner of the company OR an admin
        if ($review->user_id === $user->id) {
            return true;
        }

        if ($review->company && $review->company->user_id === $user->id) {
            return true;
        }

        if ($user && (isset($user->is_admin) && $user->is_admin)) {
            return true;
        }

        return false;
    }

    // Add policy methods for admin actions like approve/reject
    // public function approve(User $user, Review $review): bool { return $user->is_admin; }
    // public function reject(User $user, Review $review): bool { return $user->is_admin; }
}