<?php

namespace Modules\BusinessDirectory\Policies;

use App\Models\User; // Assuming your User model path is App\Models\User
use Illuminate\Auth\Access\HandlesAuthorization;
use Modules\BusinessDirectory\Entities\Product;
use Modules\BusinessDirectory\Entities\Company; // Needed to check company ownership

class ProductPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view the product.
     */
    public function view(?User $user, Product $product): bool
    {
        // Published products are publicly viewable
        if ($product->is_published) {
            return true;
        }

        // Allow the owner of the company that owns the product to view it regardless of status
        if ($user && $product->company && $product->company->user_id === $user->id) {
            return true;
        }

        // Allow administrators to view any product
        if ($user && (isset($user->is_admin) && $user->is_admin)) { // Assuming User model has an is_admin flag
            return true;
        }

        return false; // Default deny
    }

    /**
     * Determine whether the user can create products for a company.
     * This policy method receives the Product class and potentially a Company instance.
     */
    public function create(User $user, ?Company $company = null): bool
    {
        // User must be authenticated (handled by middleware)
        // If creating within a company context, user must own that company
        if ($company) {
            return $company->user_id === $user->id;
        }

        // If creating generally (e.g., from a "Post Product" link),
        // user must own at least one approved company to post products for.
        // This check might be better in the controller's create/store method
        // or a dedicated Form Request authorize method.
        // For a policy, let's assume the check is whether they *can* create if they have a valid company.
        // TODO: Add subscription/feature check here
        return $user->companies()->where('status', 'approved')->exists();
    }

    /**
     * Determine whether the user can update the product.
     */
    public function update(User $user, Product $product): bool
    {
        // User must own the company that owns the product
        return $product->company && $product->company->user_id === $user->id;
    }

    /**
     * Determine whether the user can delete the product.
     */
    public function delete(User $user, Product $product): bool
    {
        // User must own the company that owns the product
        return $product->company && $product->company->user_id === $user->id;
    }

    // A general 'manage' policy could also be used for create/update/delete if logic is the same
    // public function manage(User $user, Product $product): bool
    // {
    //     return $product->company && $product->company->user_id === $user->id;
    // }
}