<?php

namespace Modules\BusinessDirectory\Policies; // Correct namespace

use App\Models\User; // Assuming your User model path is App\Models\User
use Modules\BusinessDirectory\Entities\Company;
use Modules\BusinessDirectory\Entities\Job;
use Illuminate\Auth\Access\HandlesAuthorization;

class JobPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any jobs.
     * This is often used for index pages.
     * Public job listings are viewable by anyone.
     * @param  \App\Models\User|null  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function viewAny(?User $user)
    {
        // Anyone can attempt to view job listings (controller will filter by status 'open' for public)
        return true;
    }

    /**
     * Determine whether the user can view the specific job.
     *
     * @param  \App\Models\User|null  $user // Nullable for guests
     * @param  \Modules\BusinessDirectory\Entities\Job  $job
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function view(?User $user, Job $job)
    {
        // If the job is open, anyone can view it.
        if ($job->status === 'open') {
            return true;
        }

        // Allow authenticated users who own the company or posted the job to view non-open jobs (e.g., drafts).
        if ($user) {
            // Check if user owns the job OR if the job has a company and the user owns that company
            return $user->id === $job->user_id || ($job->company && $user->id === $job->company->user_id);
        }

        return false;
    }

    /**
     * Determine whether the user can create jobs for a specific company.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Company  $company
     * @return \Illuminate\Auth\Access\Response|bool
     * This policy method receives the Job class and the Company instance.
     */
    public function create(User $user, Company $company): bool
    {
        // User must own the company to create a job for it.
        // Add subscription/feature limit checks here or in the controller/service layer.
        return $user->id === $company->user_id;
    }

    /**
     * Determine whether the user can update the job.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Job  $job
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function update(User $user, Job $job): bool
    {
        // User who posted the job or owns the company can update it.
        return $user->id === $job->user_id || ($job->company && $user->id === $job->company->user_id);
    }

    /**
     * Determine whether the user can delete the job.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Job  $job
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function delete(User $user, Job $job): bool
    {
        // User who posted the job or owns the company can delete it.
        return $user->id === $job->user_id || ($job->company && $user->id === $job->company->user_id);
    }

    /**
     * Determine whether the user can manage the job (general permission, often used for edit/delete).
     * Your controller uses 'manage'.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Job  $job
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function manage(User $user, Job $job): bool
    {
        return $this->update($user, $job); // Delegate to update or define specific logic
    }

    // TODO: Add other policy methods like restore, forceDelete, apply, etc. if needed
    // Example for applying to a job:
    // public function apply(User $user, Job $job)
    // {
    //     return $job->status === 'open'; // Any authenticated user can apply to an open job
    // }
}