<?php

namespace Modules\BusinessDirectory\Policies; // Correct namespace
use App\Models\User; // Assuming your User model path is App\Models\User
use Modules\BusinessDirectory\Entities\Company;
use Illuminate\Auth\Access\HandlesAuthorization;

class CompanyPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any Entities.
     *
     * @param  \App\Models\User|null  $user // Nullable for guest access
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function viewAny(?User $user)
    {
        // Anyone can view the list of companies (public directory)
        return true;
    }

    /**
     * Determine whether the user can view the model.
     *
     * @param  \App\Models\User|null  $user // Nullable for guest access
     * @param  \Modules\BusinessDirectory\Entities\Company  $company
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function view(?User $user, Company $company)
    {
        // Approved companies can be viewed by anyone.
        // Pending/rejected/suspended companies might only be viewable by owner or admin.
        if ($company->status === 'approved') {
            return true;
        }

        // Allow owner or admin to view non-approved companies
        // Ensure $user is not null and check ownership or admin status
        // Assuming User model has an isAdmin() method or is_admin property
        return $user !== null && ($user->id === $company->user_id || (isset($user->is_admin) && $user->is_admin));
    }

    /**
     * Determine whether the user can create Entities.
     *
     * @param  \App\Models\User  $user  // Corrected type hint
     * @return \Illuminate\Auth\Access\Response|bool
     * Any authenticated user can attempt to create a company.
     */
    public function create(User $user): bool
    {
        // Or, you might have a specific permission: return $user->can('create companies');
        return true; // If an authenticated user reaches this, they can attempt to create.
    }

    /**
     * Determine whether the user can update the model.
     *
     * @param  \App\Models\User  $user  // Corrected type hint
     * @param  \Modules\BusinessDirectory\Entities\Company  $company
     * @return \Illuminate\Auth\Access\Response|bool
     * Only the company owner or an admin can update it.
     */
    public function update(User $user, Company $company): bool
    {
        // Assuming User model has an isAdmin() method or is_admin property
        return $user->id === $company->user_id || (isset($user->is_admin) && $user->is_admin);
    }

    /**
     * Determine whether the user can delete the model.
     *
     * @param  \App\Models\User  $user  // Corrected type hint
     * @param  \Modules\BusinessDirectory\Entities\Company  $company
     * @return \Illuminate\Auth\Access\Response|bool
     * Only the company owner or an admin can delete it.
     */
    public function delete(User $user, Company $company): bool
    {
        // Assuming User model has an isAdmin() method or is_admin property
        return $user->id === $company->user_id || (isset($user->is_admin) && $user->is_admin);
    }

    /**
     * Determine whether the user can restore the model.
     *
     * @param  \App\Models\User  $user  // Corrected type hint
     * @param  \Modules\BusinessDirectory\Entities\Company  $company
     * @return \Illuminate\Auth\Access\Response|bool
     * Only administrators can restore companies.
     */
    public function restore(User $user, Company $company): bool
    {
        return isset($user->is_admin) && $user->is_admin;
    }

    /**
     * Determine whether the user can permanently delete the model.
     *
     * @param  \App\Models\User  $user  // Corrected type hint
     * @param  \Modules\BusinessDirectory\Entities\Company  $company
     * @return \Illuminate\Auth\Access\Response|bool
     * Only administrators can permanently delete companies.
     */
    public function forceDelete(User $user, Company $company): bool
    {
        return isset($user->is_admin) && $user->is_admin;
    }
}