<?php

namespace Modules\BusinessDirectory\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Notification;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Modules\BusinessDirectory\Entities\Bid;
use Modules\BusinessDirectory\Entities\Tender;
use Illuminate\Support\Str;

class BidWithdrawnForOwnerNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public Bid $bid;
    public Tender $tender;

    /**
     * Create a new notification instance.
     *
     * @param Bid $bid The bid that was withdrawn
     */
    public function __construct(Bid $bid)
    {
        $this->bid = $bid;
        $this->tender = $bid->tender;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function via($notifiable)
    {
        return ['database', 'mail'];
    }

    /**
     * Get the array representation of the notification.
     * This is what will be stored in the 'data' column of the 'notifications' table.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function toArray($notifiable)
    {
        return [
            'tender_id' => $this->tender->id,
            'tender_title' => $this->tender->title,
            'tender_slug' => $this->tender->slug,
            'bid_id' => $this->bid->id,
            'bidder_company_id' => $this->bid->bidder_company_id,
            'bidder_company_name' => $this->bid->bidderCompany->name, // Assuming Bid model has bidderCompany relationship
            'message' => "The bid from {$this->bid->bidderCompany->name} for your tender \"{$this->tender->title}\" has been withdrawn by the bidder.",
            'link' => route('public.businessdirectory.tenders.show', $this->tender->slug) . '#bid-' . $this->bid->id,
        ];
    }

    // Optional: toMail method
    public function toMail($notifiable)
    {
        $url = route('public.businessdirectory.tenders.show', $this->tender->slug) . '#bid-' . $this->bid->id;
        return (new MailMessage)
                    ->subject('Bid Withdrawn on Your Tender: ' . Str::limit($this->tender->title, 40))
                    ->greeting('Hello ' . $notifiable->name . ',')
                    ->line("The bid submitted by {$this->bid->bidderCompany->name} for your tender \"{$this->tender->title}\" has been withdrawn.")
                    ->action('View Tender Details', $url)
                    ->line('This is for your information.');
    }
}