<?php

namespace Modules\BusinessDirectory\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Modules\BusinessDirectory\Entities\CompanyTask;
use Modules\BusinessDirectory\Entities\CompanyProject; // Import CompanyProject
use App\Models\User; // Ensure User model is imported

class UpdateCompanyTaskRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     * Authorization is handled by the CompanyTaskPolicy in the controller.
     */
    public function authorize(): bool
    {
        // The policy check is done in the controller's update method
        // $this->authorize('update', $this->route('task')); // Or 'manage' on the project
        if (!Auth::check()) {
            return false;
        }
        return true;
    }


    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $project = $this->route('project'); // Get the CompanyProject instance
        $task = $this->route('task');       // Get the CompanyTask instance
        $validAssigneeIds = collect();

        if ($project instanceof CompanyProject) {
            if ($project->team_id && $project->team) {
                $validAssigneeIds = $project->team->users()->pluck('users.id');
            } elseif ($project->company && $project->company->user) {
                $companyOwner = $project->company->user;
                $personalTeam = $companyOwner->personalTeam;
                if ($personalTeam) {
                    $validAssigneeIds = $personalTeam->users()->pluck('users.id');
                }
            }
            // Add the company owner
            if ($project->company && $project->company->user_id) {
                $validAssigneeIds->push($project->company->user_id);
            }
        }

        // Ensure the current assignee (if any) is considered valid, even if not in the current teams
        if ($task instanceof CompanyTask && $task->assignee_user_id) {
            $validAssigneeIds->push($task->assignee_user_id);
        }

        return [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'assignee_user_id' => ['nullable', 'integer', Rule::in($validAssigneeIds->unique()->toArray())],
            'due_date' => 'nullable|date',
            'priority' => ['nullable', 'string', Rule::in(array_keys(\Modules\BusinessDirectory\Entities\CompanyTask::getPriorities()))],
            'status' => ['required', 'string', Rule::in(array_keys(\Modules\BusinessDirectory\Entities\CompanyTask::getStatuses()))],
        ];
    }
}