<?php

namespace Modules\BusinessDirectory\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Modules\BusinessDirectory\Entities\Company;
use Modules\BusinessDirectory\Entities\Product;

class StoreProductRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        if (!Auth::check()) {
            return false;
        }
        // The controller's store method will perform a more specific authorization check
        // against the selected company using ProductPolicy.
        // This basic check ensures the user is logged in.
        $company = Company::find($this->input('company_id'));
        if (!$company) {
            return false; // No company selected or found
        }
        // Check if user owns the company they are trying to create a product for
        return $company->user_id === Auth::id();
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $user = Auth::user();
        $userCompanyIds = $user->companies()->where('status', 'approved')->pluck('id')->toArray();

        return [
            'company_id' => ['required', Rule::in($userCompanyIds)],
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:10000',
            'sku' => ['nullable', 'string', 'max:100', Rule::unique('bd_products', 'sku')->where(function ($query) {
                return $query->where('company_id', $this->input('company_id'));
            })],
            'price' => 'required|numeric|min:0',
            'sale_price' => 'nullable|numeric|min:0|lte:price',
            'stock_quantity' => 'nullable|integer|min:0',
            'is_published' => 'boolean', // Assuming you have a checkbox for this
            'status' => ['required', 'string', Rule::in(array_keys(Product::getStatuses()))],
            // Add rules for images, meta fields if they are part of the form
            // 'images' => 'nullable|array',
            // 'images.*' => 'image|mimes:jpeg,png,jpg,gif,svg,webp|max:2048',
        ];
    }

    /**
     * Prepare the data for validation.
     */
    protected function prepareForValidation(): void
    {
        $this->merge([
            'is_published' => $this->has('is_published'),
        ]);
    }
}