<?php

namespace Modules\BusinessDirectory\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Modules\BusinessDirectory\Entities\CompanyProject;
use App\Models\User; // Ensure User model is imported

class StoreCompanyTaskRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     * Authorization is handled by the CompanyTaskPolicy in the controller.
     */
    public function authorize(): bool
    {
        // The policy check is done in the controller's store method
        // $this->authorize('manage', $this->route('project'));
        // This basic check ensures the user is logged in.
        // The controller handles if the user can manage the specific project.
        if (!Auth::check()) {
            return false;
        }
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $project = $this->route('project'); // Get the CompanyProject instance from the route
        $validAssigneeIds = collect();

        if ($project instanceof CompanyProject) {
            if ($project->team_id && $project->team) {
                // Get members of the project's assigned team
                $validAssigneeIds = $project->team->users()->pluck('users.id'); // Ensure you pluck from the users table
            } elseif ($project->company && $project->company->user) {
                // Fallback: Get members of the company owner's personal team
                $companyOwner = $project->company->user;
                $personalTeam = $companyOwner->personalTeam;
                if ($personalTeam) {
                    $validAssigneeIds = $personalTeam->users()->pluck('users.id');
                }
            }
            // Add the company owner as a potential assignee if not already included
            if ($project->company && $project->company->user_id) {
                $validAssigneeIds->push($project->company->user_id);
            }
        }

        return [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'assignee_user_id' => ['nullable', 'integer', Rule::in($validAssigneeIds->unique()->toArray())],
            'due_date' => 'nullable|date',
            'priority' => ['nullable', 'string', Rule::in(array_keys(\Modules\BusinessDirectory\Entities\CompanyTask::getPriorities()))],
            'status' => ['required', 'string', Rule::in(array_keys(\Modules\BusinessDirectory\Entities\CompanyTask::getStatuses()))],
        ];
    }
}