<?php

namespace Modules\BusinessDirectory\Http\Requests\Admin;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Modules\BusinessDirectory\Entities\Company;

class UpdateCompanyRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        // Assuming admin middleware already protects this route.
        // Policy check for 'update' is done in the controller.
        return Auth::check() && Auth::user()->isAdmin(); // Or your specific admin check
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $company = $this->route('company'); // Get the company instance from route model binding

        return [
            'user_id' => 'required|exists:users,id',
            'company_type_id' => 'required|exists:bd_company_types,id',
            'name' => ['required', 'string', 'max:255', Rule::unique('bd_companies', 'name')->ignore($company->id)],
            'description' => 'nullable|string|max:10000',
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('bd_companies', 'email')->ignore($company->id)],
            'phone_number' => 'nullable|string|max:30',
            'website_url' => 'nullable|url|max:255',
            'whatsapp_number' => 'nullable|string|max:30',
            'address_line_1' => 'nullable|string|max:255',
            'address_line_2' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:100',
            'state_province' => 'nullable|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'country_code' => 'nullable|string|max:10',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'logo' => ['nullable', 'image', 'mimes:jpg,jpeg,png,gif,svg,webp', 'max:2048'],
            'banner' => ['nullable', 'image', 'mimes:jpg,jpeg,png,webp', 'max:5120'],
            'status' => ['required', 'string', Rule::in(array_keys(Company::getStatuses()))],
            'established_date' => 'nullable|date',
            'employee_count' => 'nullable|string|max:50',
            'annual_revenue' => 'nullable|numeric|min:0',
            'social_links.facebook' => 'nullable|url|max:255',
            'social_links.twitter' => 'nullable|url|max:255',
            'social_links.linkedin' => 'nullable|url|max:255',
            'social_links.instagram' => 'nullable|url|max:255',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:1000',
            'meta_keywords' => 'nullable|string|max:255',
            'rejection_reason' => 'nullable|string|max:1000|required_if:status,rejected',
            'suspension_reason' => 'nullable|string|max:1000|required_if:status,suspended',
            'is_featured' => 'nullable|boolean',
            'is_verified' => 'nullable|boolean',
            'is_ecommerce' => 'nullable|boolean',
        ];
    }
}