<?php

namespace Modules\BusinessDirectory\Http\Requests\Admin;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Modules\BusinessDirectory\Entities\Job;

class StoreJobRequest extends FormRequest
{
    public function authorize(): bool
    {
        return Auth::check() && Auth::user()->isAdmin();
    }

    public function rules(): array
    {
        return [
            'company_id' => 'required|exists:bd_companies,id',
            'user_id' => 'nullable|exists:users,id', // Admin can assign, else defaults to company owner or logged-in admin
            'title' => ['required', 'string', 'max:255', Rule::unique('bd_jobs', 'title')],
            'description' => 'required|string|max:20000',
            'job_type' => ['required', 'string', Rule::in(array_keys(Job::getJobTypes()))],
            'location' => 'nullable|string|max:255',
            'salary_min' => 'nullable|numeric|min:0|lte:salary_max',
            'salary_max' => 'nullable|numeric|min:0|gte:salary_min',
            'currency' => 'nullable|string|max:10',
            'experience_level' => 'nullable|string|max:100',
            'education_level' => 'nullable|string|max:100',
            'skills_required' => 'nullable|array',
            'skills_required.*' => 'string|max:100',
            'application_deadline' => 'nullable|date|after_or_equal:today',
            'status' => ['required', 'string', Rule::in(array_keys(Job::getStatuses()))],
            'is_remote' => 'nullable|boolean',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:1000',
            'meta_keywords' => 'nullable|string|max:255',
        ];
    }

    protected function prepareForValidation()
    {
        $this->merge([
            'is_remote' => $this->boolean('is_remote'),
            'skills_required' => is_string($this->skills_required) ?
                                array_map('trim', explode(',', $this->skills_required)) :
                                $this->skills_required,
        ]);

        // Default user_id if not provided
        if (!$this->user_id && $this->company_id) {
            $company = \Modules\BusinessDirectory\Entities\Company::find($this->company_id);
            $this->merge(['user_id' => $company?->user_id ?? auth()->id()]);
        } elseif (!$this->user_id) {
            $this->merge(['user_id' => auth()->id()]);
        }
    }
}