<?php

namespace Modules\BusinessDirectory\Http\Controllers\Frontend;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Modules\BusinessDirectory\Entities\Company;
use Modules\BusinessDirectory\Entities\Review;
// Assuming you have a ReviewPolicy
// use Modules\BusinessDirectory\Policies\ReviewPolicy;
// use App\Services\SubscriptionService; // Example

class ReviewController extends Controller
{
    use AuthorizesRequests;

    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Store a newly created review in storage.
     *
     * @param Request $request
     * @param Company $company
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request, Company $company)
    {
        $this->authorize('create', [Review::class, $company]);

        $request->validate([
            'rating' => 'required|integer|min:1|max:5',
            'title' => 'nullable|string|max:255',
            'comment' => 'required|string|max:5000',
        ]);

        $user = Auth::user();

        $existingReview = Review::where('company_id', $company->id)
                                ->where('user_id', $user->id)
                                ->first();

        if ($existingReview) {
            return back()->with('error', 'You have already submitted a review for this company.');
        }

        // Eligibility Check: Implement logic to ensure the user is eligible to review.
        // For example, has the user purchased a product or completed an order from this company?

        $review = Review::create([ // Assign to $review to pass to the event
            'company_id' => $company->id,
            'user_id' => $user->id,
            'rating' => $request->rating,
            'title' => $request->title,
            'comment' => $request->comment,
            'is_approved' => false,
        ]);

        // Notify admin about the new review for approval
        event(new \Modules\BusinessDirectory\Events\NewReviewSubmittedForApproval($review));

        return back()->with('success', 'Your review has been submitted and is pending approval. Thank you!');
    }

    /**
     * Display a listing of the reviews submitted by the authenticated user.
     *
     * @param Request $request
     * @return Renderable
     */
    public function myReviews(Request $request)
    {
        $user = Auth::user();
        $reviews = Review::where('user_id', $user->id)
                         ->with('company:id,name,slug') // Eager load company details
                         ->latest()
                         ->paginate(10);

        return view('businessdirectory::frontend.reviews.my_reviews', compact('reviews'));
    }
}
