<?php

namespace Modules\BusinessDirectory\Http\Controllers\Frontend;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Modules\BusinessDirectory\Entities\Job;
use Modules\BusinessDirectory\Entities\JobApplication;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class JobApplicationController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Store a newly created job application.
     * @param Request $request
     * @param Job $job
     */
    public function store(Request $request, Job $job)
    {
        if ($job->status !== 'open') {
            return back()->with('error', 'This job is no longer accepting applications.');
        }

        $request->validate([
            'resume' => 'required|file|mimes:pdf,doc,docx|max:2048', // Max 2MB
            'cover_letter' => 'nullable|string|max:5000',
        ]);

        // Check if user has already applied
        $existingApplication = JobApplication::where('job_id', $job->id)
                                             ->where('user_id', Auth::id())
                                             ->first();

        if ($existingApplication) {
            return back()->with('error', 'You have already applied for this job.');
        }

        $resumePath = null;
        if ($request->hasFile('resume')) {
            $resumePath = $request->file('resume')->store('job_applications/resumes/' . Auth::id(), 'public');
        }

        $application = JobApplication::create([ // Assign to $application
            'job_id' => $job->id,
            'user_id' => Auth::id(),
            'resume_path' => $resumePath,
            'cover_letter' => $request->cover_letter,
            'status' => 'applied',
            'applied_at' => now(),
        ]);

        // Notify company owner about the new application
        event(new \Modules\BusinessDirectory\Events\NewJobApplicationSubmitted($application));

        return redirect()->route('frontend.businessdirectory.jobs.show', $job->slug)
            ->with('success', 'Your application has been submitted successfully!');
    }
    /**
     * Display a listing of the user's job applications.
     * @param Request $request
     */
    public function myApplications(Request $request)
    {
        $user = Auth::user();
        $query = JobApplication::with('job.company')
                           ->where('user_id', $user->id);

        // Apply Search Filter
        if ($request->filled('search')) {
            $searchTerm = $request->input('search');
            $query->whereHas('job', function ($q) use ($searchTerm) {
                $q->where('title', 'like', "%{$searchTerm}%")
                  ->orWhereHas('company', function ($cq) use ($searchTerm) {
                      $cq->where('name', 'like', "%{$searchTerm}%");
                  });
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->input('status'));
        }
        $applications = $query
            ->latest('applied_at')
            ->paginate(10);

        return view('businessdirectory::frontend.jobs.my_applications', compact('applications'));
    }
}