<?php

namespace Modules\BusinessDirectory\Http\Controllers\Frontend;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Support\Facades\Auth;
use Modules\BusinessDirectory\Entities\CompanyProject;
use Modules\BusinessDirectory\Entities\CompanyTask;
use App\Models\User; // Corrected User model path
use Modules\BusinessDirectory\Http\Requests\StoreCompanyTaskRequest; // Import Form Requests
use Modules\BusinessDirectory\Http\Requests\UpdateCompanyTaskRequest; // Import Form Requests
use Modules\BusinessDirectory\Events\CompanyTaskStatusUpdated; // Import the new event

class CompanyTaskController extends Controller
{
    use AuthorizesRequests;

    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Show the form for creating a new task for a user's company project.
     */
    public function create(CompanyProject $project)
    {   
        $this->authorize('manage', $project); // User must be able to manage the parent project
        
        $assignees = collect();
        if ($project->team_id && $project->team) {
            // Get members of the project's assigned team
            $assignees = $project->team->users()->orderBy('name')->pluck('name', 'id');
        } elseif ($project->company && $project->company->user) {
            // Fallback: Get members of the company owner's personal team
            $companyOwner = $project->company->user;
            $personalTeam = $companyOwner->personalTeam; // Assumes personalTeam relationship exists on User model
            if ($personalTeam) {
                $assignees = $personalTeam->users()->orderBy('name')->pluck('name', 'id');
            }
        }
        // If still no assignees, you might want to add the company owner or project creator as a last resort
        if ($assignees->isEmpty() && $project->company && $project->company->user) {
             $assignees->put($project->company->user->id, $project->company->user->name . ' (Company Owner)');
        }

        $task_statuses = CompanyTask::getStatuses(); // Define task_statuses
        $priorities = CompanyTask::getPriorities(); // Use the model's method for consistency
        return view('businessdirectory::frontend.tasks.my_tasks_create', compact('project', 'assignees', 'task_statuses', 'priorities'));
    }

    /**
     * Store a newly created task.
     */
    public function store(StoreCompanyTaskRequest $request, CompanyProject $project) // Use Form Request
    {   
        $this->authorize('manage', $project);

        $taskData = $request->validated(); // Use validated data
        $taskData['project_id'] = $project->id;
        $taskData['user_id'] = Auth::id(); // Add the authenticated user's ID
        CompanyTask::create($taskData);

        return redirect()->route('frontend.businessdirectory.my-projects.show', $project->slug)
            ->with('success', 'Task added successfully.');
    }

    /**
     * Show the form for editing a task.
     */
    public function edit(CompanyProject $project, CompanyTask $task)
    {   
        $this->authorize('manage', $project);

        $assignees = collect();
        if ($project->team_id && $project->team) {
            $assignees = $project->team->users()->orderBy('name')->pluck('name', 'id');
        } elseif ($project->company && $project->company->user) {
            $companyOwner = $project->company->user;
            $personalTeam = $companyOwner->personalTeam;
            if ($personalTeam) {
                $assignees = $personalTeam->users()->orderBy('name')->pluck('name', 'id');
            }
        }
        // Ensure current assignee is in the list if they are not part of the default teams
        if ($task->assignee_user_id && !$assignees->has($task->assignee_user_id) && $task->assignee) {
            $assignees->put($task->assignee_user_id, $task->assignee->name);
        }

        $task_statuses = CompanyTask::getStatuses(); // Define task_statuses
        $priorities = CompanyTask::getPriorities(); // Use the model's method for consistency
        return view('businessdirectory::frontend.tasks.my_tasks_edit', compact('project', 'task', 'assignees', 'task_statuses', 'priorities'));
    }

    /**
     * Update the specified task.
     */
    public function update(UpdateCompanyTaskRequest $request, CompanyProject $project, CompanyTask $task) // Use FormRequest
    {   
        $this->authorize('manage', $project);

        $validatedData = $request->validated();
        $oldStatus = $task->status;

        $task->update($validatedData);

        if ($oldStatus !== $task->status) {
            event(new CompanyTaskStatusUpdated($task, $oldStatus, $task->status));
        }

        return redirect()->route('frontend.businessdirectory.my-projects.show', $project->slug)
            ->with('success', 'Task updated successfully.');
    }

    /**
     * Remove the specified task.
     */
    public function destroy(CompanyProject $project, CompanyTask $task)
    {
        $this->authorize('manage', $project); // Policy check

        $task->delete();

        return redirect()->route('frontend.businessdirectory.my-projects.show', $project->slug)
            ->with('success', 'Task deleted successfully.');
    }
}