<?php

namespace Modules\BusinessDirectory\Http\Controllers\Frontend;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Modules\BusinessDirectory\Entities\Company;
use Modules\BusinessDirectory\Entities\CompanyProject;
use Modules\BusinessDirectory\Http\Requests\StoreCompanyProjectRequest; // Create this if you don't have it
use Modules\BusinessDirectory\Http\Requests\UpdateCompanyProjectRequest; // Create this if you don't have it

class CompanyProjectController extends Controller
{
    use AuthorizesRequests;

    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     * @return Renderable
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        // Get projects for companies owned by the user
        $companyIds = $user->companies()->pluck('id');
        $projects = CompanyProject::whereIn('company_id', $companyIds)
            ->with('company:id,name,slug')
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('businessdirectory::frontend.projects.my_projects_index', compact('projects'));
    }

    /**
     * Show the form for creating a new resource.
     * @return Renderable|\Illuminate\Http\RedirectResponse
     */
    public function create()
    {
        $user = Auth::user();
        $companies = $user->companies()->where('status', 'approved')->orderBy('name')->pluck('name', 'id');

        if ($companies->isEmpty()) {
            return redirect()->route('frontend.businessdirectory.my-companies.index')
                             ->with('warning', 'You need an approved company to create a project.');
        }

        // Authorize against the first available company for showing the form.
        // The actual company will be selected in the form.
        // The store method will authorize against the selected company.
        $firstCompany = $user->companies()->where('status', 'approved')->first();
        if (!$firstCompany) { // Should be caught by isEmpty above, but as a safeguard
             abort(403, 'No eligible company found to create a project for.');
        }
        
        // This is the line (approx. 81 from your error) that needs the Company instance
        $this->authorize('create', [CompanyProject::class, $firstCompany]);

        $project = new CompanyProject(); // For form model binding
        $teams = Auth::user()->ownedTeams()->orderBy('name')->pluck('name', 'id'); // Get teams owned by the current user
        return view('businessdirectory::frontend.projects.my_projects_create', compact('project', 'companies', 'teams'));
    }

    /**
     * Store a newly created resource in storage.
     * @param StoreCompanyProjectRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(StoreCompanyProjectRequest $request) // Use a FormRequest
    {
        $validatedData = $request->validated();
        $company = Company::findOrFail($validatedData['company_id']);

        $this->authorize('create', [CompanyProject::class, $company]);

        $project = $company->projects()->create($validatedData + ['user_id' => Auth::id()]);

        return redirect()->route('frontend.businessdirectory.my-projects.index')
                         ->with('success', 'Project created successfully.');
    }

    /**
     * Show the specified resource.
     * @param CompanyProject $project
     * @return Renderable
     */
    public function show(CompanyProject $project)
    {
        $this->authorize('view', $project);
        $project->load('company', 'tasks');
        return view('businessdirectory::frontend.projects.my_projects_show', compact('project'));
    }

    /**
     * Show the form for editing the specified resource.
     * @param CompanyProject $project
     * @return Renderable
     */
    public function edit(CompanyProject $project)
    {
        $this->authorize('update', $project);
        $user = Auth::user();
        $companies = $user->companies()->where('status', 'approved')->orderBy('name')->pluck('name', 'id');
        $teams = Auth::user()->ownedTeams()->orderBy('name')->pluck('name', 'id'); // Get teams owned by the current user
        return view('businessdirectory::frontend.projects.my_projects_edit', compact('project', 'companies', 'teams'));
    }

    /**
     * Update the specified resource in storage.
     * @param UpdateCompanyProjectRequest $request
     * @param CompanyProject $project
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(UpdateCompanyProjectRequest $request, CompanyProject $project)
    {
        $this->authorize('update', $project);
        $validatedData = $request->validated();
        $project->update($validatedData);

        return redirect()->route('frontend.businessdirectory.my-projects.index')
                         ->with('success', 'Project updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     * @param CompanyProject $project
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(CompanyProject $project)
    {
        $this->authorize('delete', $project);
        $project->delete();

        return redirect()->route('frontend.businessdirectory.my-projects.index')
                         ->with('success', 'Project deleted successfully.');
    }
}