<?php

namespace Modules\BusinessDirectory\Http\Controllers\Frontend;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Modules\BusinessDirectory\Entities\Cart;
use Modules\BusinessDirectory\Entities\CartItem;
use Modules\BusinessDirectory\Entities\Product;

class CartController extends Controller
{
    /**
     * Get the current user's cart or create one if it doesn't exist.
     */    
    protected function getOrCreateCart()
    {
        if (Auth::check()) {
            return Cart::firstOrCreate(['user_id' => Auth::id()]);
        } else {
            $sessionId = Session::getId();
            return Cart::firstOrCreate(['session_id' => $sessionId]);
        }
    }

    /**
     * Display the shopping cart.
     */
    public function index()
    {
        $cart = $this->getOrCreateCart();
        $cart->load('items.product.company'); // Eager load product and its company

        return view('businessdirectory::frontend.cart.index', compact('cart'));
    }

    /**
     * Add an item to the cart.
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function add(Request $request, Product $product)
    {
        $request->validate([
            'quantity' => 'required|integer|min:1',
        ]);

        if ($product->status !== 'published' || $product->stock_quantity < $request->quantity) {
            return back()->with('error', 'Product is unavailable or insufficient stock.');
        }

        $cart = $this->getOrCreateCart();

        $cartItem = $cart->items()->where('product_id', $product->id)->first();
        if ($cartItem) {
            $newQuantity = $cartItem->quantity + $request->quantity;
            if ($product->stock_quantity < $newQuantity) {
                 return back()->with('error', 'Cannot add requested quantity. Insufficient stock.');
            }
            $cartItem->quantity = $newQuantity;
            $cartItem->save();
        } else {
            // Add new item
            $cart->items()->create([
                'product_id' => $product->id,
                'quantity' => $request->quantity,
            ]);
        }

        return redirect()->route('frontend.businessdirectory.cart.index')->with('success', 'Product added to cart!');
    }

    /**
     * Update the quantity of an item in the cart.
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, CartItem $cartItem)
    {
        $cart = $this->getOrCreateCart();
        // Ensure the cart item belongs to the current cart
        if ($cartItem->cart_id !== $cart->id) {
            return back()->with('error', 'Invalid cart item.');
        }

        $request->validate([
            'quantity' => 'required|integer|min:1',
        ]);
        if ($cartItem->product->stock_quantity < $request->quantity) {
            return back()->with('error', 'Cannot update to requested quantity. Insufficient stock.');
        }

        $cartItem->update(['quantity' => $request->quantity]);

        return redirect()->route('frontend.businessdirectory.cart.index')->with('success', 'Cart updated.');
    }

    /**
     * Remove an item from the cart.
     * @return \Illuminate\Http\RedirectResponse
     */
    public function remove(CartItem $cartItem)
    {
        $cart = $this->getOrCreateCart();
        if ($cartItem->cart_id !== $cart->id) {
            return back()->with('error', 'Invalid cart item.');
        }

        $cartItem->delete();
        return redirect()->route('frontend.businessdirectory.cart.index')->with('success', 'Item removed from cart.');
    }

    /**
     * Clear all items from the cart.
     * @return \Illuminate\Http\RedirectResponse
     */
    public function clear()
    {
        $cart = $this->getOrCreateCart();
        $cart->items()->delete();
        return redirect()->route('frontend.businessdirectory.cart.index')->with('success', 'Cart cleared.');
    }
}