<?php

namespace Modules\BusinessDirectory\Http\Controllers\Admin;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Modules\BusinessDirectory\Entities\Review;

class ReviewController extends Controller
{
    public function __construct()
    {
        // Policy authorization per method.
    }
    /**
     * Display a listing of the resource.
     * @return Renderable
     */
    public function index(Request $request)
    {
        $this->authorize('viewAny', Review::class);
        $query = Review::with(['company', 'user'])->latest();

        if ($request->filled('search')) {
            $searchTerm = $request->search;
            $query->where('title', 'like', "%{$searchTerm}%")
                  ->orWhere('comment', 'like', "%{$searchTerm}%")
                  ->orWhereHas('company', fn($q) => $q->where('name', 'like', "%{$searchTerm}%"))
                  ->orWhereHas('user', fn($q) => $q->where('name', 'like', "%{$searchTerm}%"));
        }

        if ($request->filled('status_filter')) {
            $query->where('is_approved', $request->status_filter === '1'); // '1' for approved, '0' for pending
        }

        $reviews = $query->paginate(15)->withQueryString();
        // $statuses_filter = ['1' => 'Approved', '0' => 'Pending']; // For the filter dropdown

        return view('businessdirectory::admin.reviews.index', compact('reviews' /*, 'statuses_filter'*/));
    }

    /**
     * Show the specified resource.
     * @param Review $review
     * @return Renderable
     */
    public function show(Review $review)
    {
        $this->authorize('view', $review);
        $review->load(['company', 'user']);
        return view('businessdirectory::admin.reviews.show', compact('review'));
    }

    /**
     * Update the specified resource's approval status in storage.
     * @param Request $request
     * @param Review $review
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateStatus(Request $request, Review $review)
    {
        $this->authorize('updateStatus', $review); // Or 'update' if admin role handles this via before()
        $request->validate([
            'is_approved' => 'required|boolean',
        ]);

        $review->is_approved = $request->is_approved;
        $review->save();

        // Notify user if needed
        // ReviewStatusChangedNotification::dispatch($review->user, $review);

        return redirect()->back()->with('success', 'Review status updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     * @param Review $review
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Review $review)
    {
        $this->authorize('delete', $review);
        try {
            $review->delete();
            return redirect()->route('admin.businessdirectory.reviews.index')->with('success', 'Review deleted successfully.');
        } catch (\Exception $e) {
            return redirect()->route('admin.businessdirectory.reviews.index')->with('error', 'Failed to delete review.');
        }
    }
}