<?php

namespace Modules\BusinessDirectory\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\BusinessDirectory\Entities\Company;
use Modules\BusinessDirectory\Entities\Product;
use Modules\BusinessDirectory\Http\Requests\Admin\StoreProductRequest;
use Modules\BusinessDirectory\Http\Requests\Admin\UpdateProductRequest;
use Illuminate\Support\Facades\Storage;
// Assuming you might have Product Categories
// use Modules\BusinessDirectory\Entities\ProductCategory;

class ProductController extends Controller
{
    public function __construct()
    {
        // Policy authorization per method.
    }

    public function index(Request $request, Company $company)
    {
        $this->authorize('viewAny', [Product::class, $company]);
        $query = $company->products()->latest();

        if ($request->filled('search')) {
            $query->where('name', 'like', "%{$request->search}%")
                  ->orWhere('sku', 'like', "%{$request->search}%");
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $products = $query->paginate(15);
        $statuses = Product::getStatuses(); // Assuming Product model has getStatuses()

        return view('businessdirectory::admin.products.index', compact('company', 'products', 'statuses'));
    }

    public function create(Company $company)
    {
        $this->authorize('create', [Product::class, $company]);
        $statuses = Product::getStatuses();
        // $categories = ProductCategory::orderBy('name')->pluck('name', 'id'); // If you have categories
        return view('businessdirectory::admin.products.create', compact('company', 'statuses' /*, 'categories'*/));
    }

    public function store(StoreProductRequest $request, Company $company)
    {
        $this->authorize('create', [Product::class, $company]);
        $validatedData = $request->validated();

        $validatedData['company_id'] = $company->id;
        // Admin might set the user_id or it defaults to company owner
        $validatedData['user_id'] = $validatedData['user_id'] ?? $company->user_id;

        if ($request->hasFile('featured_image')) {
            $validatedData['featured_image'] = $request->file('featured_image')->store('product_images', 'public');
        }

        // Handle gallery images if your form supports multiple uploads
        if ($request->hasFile('gallery_images')) {
            $galleryPaths = [];
            foreach ($request->file('gallery_images') as $file) {
                $galleryPaths[] = $file->store('product_gallery', 'public');
            }
            $validatedData['gallery_images'] = $galleryPaths; // Assuming 'gallery_images' is a JSON column
        }

        Product::create($validatedData);

        return redirect()->route('admin.businessdirectory.companies.products.index', $company->id)
                         ->with('success', 'Product created successfully.');
    }

    public function edit(Company $company, Product $product)
    {
        $this->authorize('update', $product);
        $statuses = Product::getStatuses();
        // $categories = ProductCategory::orderBy('name')->pluck('name', 'id');
        return view('businessdirectory::admin.products.edit', compact('company', 'product', 'statuses' /*, 'categories'*/));
    }

    public function update(UpdateProductRequest $request, Company $company, Product $product)
    {
        $this->authorize('update', $product);
        $validatedData = $request->validated();

        if ($request->hasFile('featured_image')) {
            if ($product->featured_image) {
                Storage::disk('public')->delete($product->featured_image);
            }
            $validatedData['featured_image'] = $request->file('featured_image')->store('product_images', 'public');
        }

        // Handle gallery image updates (more complex: add new, remove old)
        // For simplicity, this example just replaces if new ones are uploaded.
        // A more robust solution would handle individual image additions/deletions.
        if ($request->hasFile('gallery_images')) {
            // Delete old gallery images if necessary
            if (is_array($product->gallery_images)) {
                foreach ($product->gallery_images as $oldImage) {
                    Storage::disk('public')->delete($oldImage);
                }
            }
            $galleryPaths = [];
            foreach ($request->file('gallery_images') as $file) {
                $galleryPaths[] = $file->store('product_gallery', 'public');
            }
            $validatedData['gallery_images'] = $galleryPaths;
        } elseif ($request->boolean('remove_gallery_images')) { // Example: a checkbox to clear gallery
             if (is_array($product->gallery_images)) {
                foreach ($product->gallery_images as $oldImage) {
                    Storage::disk('public')->delete($oldImage);
                }
            }
            $validatedData['gallery_images'] = null;
        }


        $product->update($validatedData);

        return redirect()->route('admin.businessdirectory.companies.products.index', $company->id)
                         ->with('success', 'Product updated successfully.');
    }

    public function destroy(Company $company, Product $product)
    {
        $this->authorize('delete', $product);

        if ($product->featured_image) {
            Storage::disk('public')->delete($product->featured_image);
        }
        if (is_array($product->gallery_images)) {
            foreach ($product->gallery_images as $galleryImage) {
                Storage::disk('public')->delete($galleryImage);
            }
        }
        // Consider if orders are associated with this product before deleting.
        $product->delete();

        return redirect()->route('admin.businessdirectory.companies.products.index', $company->id)
                         ->with('success', 'Product deleted successfully.');
    }
}