<?php

namespace Modules\BusinessDirectory\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\BusinessDirectory\Entities\Category;
use Modules\BusinessDirectory\Http\Requests\Admin\StoreCategoryRequest;
use Modules\BusinessDirectory\Http\Requests\Admin\UpdateCategoryRequest;
use Illuminate\Support\Facades\Storage;

class CategoryController extends Controller
{
    public function __construct()
    {
        $this->authorizeResource(Category::class, 'category');
    }

    public function index(Request $request)
    {
        $query = Category::with('parent')->withCount(['companies', 'products', 'tenders', 'jobs'])->latest();
        if ($request->filled('search')) {
            $query->where('name', 'like', "%{$request->search}%");
        }
        $categories = $query->paginate(15);
        return view('businessdirectory::admin.categories.index', compact('categories'));
    }

    public function create()
    {
        $parentCategories = Category::whereNull('parent_id')->orderBy('name')->pluck('name', 'id');
        return view('businessdirectory::admin.categories.create', compact('parentCategories'));
    }

    public function store(StoreCategoryRequest $request)
    {
        $validatedData = $request->validated();
        if ($request->hasFile('image_path')) {
            $validatedData['image_path'] = $request->file('image_path')->store('category_images', 'public');
        }
        Category::create($validatedData);
        return redirect()->route('admin.businessdirectory.categories.index')->with('success', 'Category created successfully.');
    }

    public function edit(Category $category)
    {
        $parentCategories = Category::whereNull('parent_id')->where('id', '!=', $category->id)->orderBy('name')->pluck('name', 'id');
        return view('businessdirectory::admin.categories.edit', compact('category', 'parentCategories'));
    }

    public function update(UpdateCategoryRequest $request, Category $category)
    {
        $validatedData = $request->validated();
        if ($request->hasFile('image_path')) {
            if ($category->image_path) {
                Storage::disk('public')->delete($category->image_path);
            }
            $validatedData['image_path'] = $request->file('image_path')->store('category_images', 'public');
        } elseif ($request->boolean('remove_image_path')) {
             if ($category->image_path) {
                Storage::disk('public')->delete($category->image_path);
            }
            $validatedData['image_path'] = null;
        }

        $category->update($validatedData);
        return redirect()->route('admin.businessdirectory.categories.index')->with('success', 'Category updated successfully.');
    }

    public function destroy(Category $category)
    {
        // Add check for associated items before deletion if necessary
        if ($category->children()->count() > 0 || $category->companies()->count() > 0 /* || etc. */) {
             return redirect()->route('admin.businessdirectory.categories.index')->with('error', 'Cannot delete category. It has child categories or associated items.');
        }
        if ($category->image_path) {
            Storage::disk('public')->delete($category->image_path);
        }
        $category->delete();
        return redirect()->route('admin.businessdirectory.categories.index')->with('success', 'Category deleted successfully.');
    }
}
