<?php

namespace Modules\BusinessDirectory\Helpers;

use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\DB;
use Modules\BusinessDirectory\Entities\Company;
use Modules\BusinessDirectory\Entities\Product;
use Modules\BusinessDirectory\Entities\Tender;
use Modules\BusinessDirectory\Entities\Job;

class FeaturedContentHelper
{
    /**
     * Get featured items with a fallback to other items if the limit is not met.
     *
     * @param string $modelClass The Eloquent model class name.
     * @param int $limit The desired number of items.
     * @param callable $featuredQueryModifier A closure to modify the query for featured items.
     * @param callable $fallbackQueryModifier A closure to modify the query for fallback items.
     * @param array $withRelations Eager loading relations.
     * @param bool $randomOrderFeatured Whether to order featured items randomly.
     * @param bool $randomOrderFallback Whether to order fallback items randomly.
     * @param string $defaultOrderBy Default order by column for fallback if not random.
     * @param string $defaultOrderDir Default order direction for fallback if not random.
     * @return Collection
     */
    public static function getItems(
        string $modelClass,
        int $limit,
        callable $featuredQueryModifier,
        callable $fallbackQueryModifier,
        array $withRelations = [],
        bool $randomOrderFeatured = true,
        bool $randomOrderFallback = true,
        string $defaultOrderBy = 'created_at',
        string $defaultOrderDir = 'desc'
    ): Collection {
        if ($limit <= 0) {
            return new Collection();
        }

        $featuredQuery = $modelClass::query();
        $featuredQueryModifier($featuredQuery); // Apply specific featured conditions

        if (!empty($withRelations)) {
            $featuredQuery->with($withRelations);
        }

        if ($randomOrderFeatured) {
            $featuredItems = $featuredQuery->inRandomOrder()->take($limit)->get();
        } else {
            // If not random, assume the modifier handles ordering or use a default
            $featuredItems = $featuredQuery->take($limit)->get();
        }

        if ($featuredItems->count() < $limit) {
            $fallbackQuery = $modelClass::query();
            $fallbackQueryModifier($fallbackQuery); // Apply specific fallback conditions

            if (!empty($withRelations)) {
                $fallbackQuery->with($withRelations);
            }

            $additionalItemsNeeded = $limit - $featuredItems->count();
            $fallbackQuery->whereNotIn('id', $featuredItems->pluck('id')->toArray());

            if ($randomOrderFallback) {
                $additionalItems = $fallbackQuery->inRandomOrder()->take($additionalItemsNeeded)->get();
            } else {
                $additionalItems = $fallbackQuery->orderBy($defaultOrderBy, $defaultOrderDir)->take($additionalItemsNeeded)->get();
            }
            $featuredItems = $featuredItems->merge($additionalItems);
        }

        return $featuredItems;
    }

    public static function getFeaturedCompanies(int $limit): Collection
    {
        return self::getItems(
            Company::class,
            $limit,
            fn($q) => $q->approved()->featured(),
            fn($q) => $q->approved()->where('is_featured', false)
        );
    }

    public static function getFeaturedProducts(int $limit): Collection
    {
        return self::getItems(
            Product::class,
            $limit,
            fn($q) => $q->published()->whereHas('company', fn($cq) => $cq->approved()->featured()),
            fn($q) => $q->published()->whereHas('company', fn($cq) => $cq->approved()), // Fallback to any approved company's products
            ['company:id,name,slug']
        );
    }

    public static function getLatestTenders(int $limit): Collection
    {
        return self::getItems(
            Tender::class,
            $limit,
            fn($q) => $q->open()->whereHas('company', fn($cq) => $cq->approved()->featured())->orderBy('deadline', 'asc'),
            fn($q) => $q->open()->whereHas('company', fn($cq) => $cq->approved())->orderBy('deadline', 'asc'),
            ['company:id,name,slug'],
            false, false, 'deadline', 'asc' // Not random, order by deadline
        );
    }

    public static function getLatestJobs(int $limit): Collection
    {
        return self::getItems(
            Job::class,
            $limit,
            fn($q) => $q->open()->whereHas('company', fn($cq) => $cq->approved()->featured())->orderBy('created_at', 'desc'),
            fn($q) => $q->open()->whereHas('company', fn($cq) => $cq->approved())->orderBy('created_at', 'desc'),
            ['company:id,name,slug,logo'],
            false, false, 'created_at', 'desc' // Not random, order by creation
        );
    }
}