<?php

namespace Modules\BusinessDirectory\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;
use App\Models\User; // Standardized User model path
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphMany; // Added for attachments
use Illuminate\Database\Eloquent\Relations\HasMany;
use Modules\BusinessDirectory\Entities\Category; // Import the correct Category model

class Tender extends Model
{
    use HasFactory, SoftDeletes, HasSlug;

    protected $table = 'bd_tenders';

    protected $fillable = [
        'company_id', // The company posting the tender
        'category_id',
        'user_id',    // The user who created the tender (can be company owner or other authorized user)
        'title',
        'slug',
        'description',
        'budget_min',
        'budget_max',
        'currency',
        'category_id', // Foreign key for TenderCategory
        'type', // e.g., fixed_price, hourly_rate, milestone_based
        'location',
        'deadline',
        'status', // e.g., draft, open, evaluation, awarded, closed, cancelled
        'attachments', // JSON
        'views_count',
    ];

    protected $casts = [
        'deadline' => 'datetime',
        'budget_min' => 'decimal:2',
        'budget_max' => 'decimal:2',
        'attachments' => 'array',
        'views_count' => 'integer',
    ];

    // Define status constants
    public const STATUS_DRAFT = 'draft';
    public const STATUS_OPEN = 'open';
    public const STATUS_EVALUATION = 'evaluation';
    public const STATUS_AWARDED = 'awarded';
    public const STATUS_CLOSED = 'closed';
    public const STATUS_CANCELLED = 'cancelled';

    /**
     * Get the company that posted the tender.
     */
    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * Get the user who created the tender.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * Get the bids for the tender.
     */
    public function bids(): HasMany
    {
        return $this->hasMany(Bid::class);
    }

    /**
     * Get all of the tender's additional attachments (MorphMany relationship).
     */
    public function additionalAttachments(): MorphMany // Renamed relationship
    {
        return $this->morphMany(Attachment::class, 'attachable');
    }

    /**
     * Get the category of the tender.
     * Assumes a TenderCategory model exists.
     */
    public function category(): BelongsTo
    {
        // Use the existing general Category model
        return $this->belongsTo(Category::class, 'category_id');
    }

    /**
     * Get the options for generating the slug.
     */
    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()
            ->generateSlugsFrom('title')
            ->doNotGenerateSlugsOnUpdate() // Prevent slug from changing if title is updated
            ->saveSlugsTo('slug');
    }

    protected static function newFactory()
    {
        return \Modules\BusinessDirectory\Database\factories\TenderFactory::new();
    }

    /**
     * Get the defined statuses for a tender.
     *
     * @return array
     */
    public static function getStatuses(): array
    {
        return [
            'draft' => 'Draft',
            'open' => 'Open for Bids',
            'evaluation' => 'Under Evaluation',
            'awarded' => 'Awarded',
            'closed' => 'Closed',
            'cancelled' => 'Cancelled',
        ];
    }

    /**
     * Get the defined types for a tender.
     *
     * @return array
     */
    public static function getTenderTypes(): array
    {
        return [
            'fixed_price' => 'Fixed Price',
            'hourly_rate' => 'Hourly Rate',
            'milestone_based' => 'Milestone-based',
            // Add other types as needed, e.g., 'open_ended', 'consultation'
        ];
    }

    /**
     * Scope a query to only include open tenders.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeOpen($query)
    {
        return $query->where('status', self::STATUS_OPEN)->where('deadline', '>=', now());
    }
}
