<?php

namespace Modules\BusinessDirectory\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Product extends Model
{
    use HasFactory, SoftDeletes, HasSlug;

    public const STATUS_DRAFT = 'draft';
    public const STATUS_PUBLISHED = 'published';
    public const STATUS_ARCHIVED = 'archived';

    protected $table = 'bd_products';

    protected $fillable = [
        'company_id',
        'name',
        'category_id',
        'slug',
        'description',
        'sku',
        'price',
        'sale_price',
        'stock_quantity',
        'is_featured',
        'status', // e.g., draft, published, archived, out_of_stock
        'images', // JSON
        'meta_title',
        'meta_description',
        'meta_keywords',
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'sale_price' => 'decimal:2',
        'stock_quantity' => 'integer',
        'is_featured' => 'boolean',
        'images' => 'array',
    ];

    // protected $dates = ['deleted_at']; // Already handled by SoftDeletes trait

    /**
     * Get the options for generating the slug.
     */
    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()
            ->generateSlugsFrom('name')
            ->saveSlugsTo('slug');
    }

    /**
     * Get the company that owns this product.
     */
    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class);
    }

    public function cartItems(): HasMany
    {
        return $this->hasMany(CartItem::class);
    }

    public function orderItems(): HasMany
    {
        return $this->hasMany(OrderItem::class);
    }

    protected static function newFactory()
    {
        return \Modules\BusinessDirectory\Database\factories\ProductFactory::new();
    }

    /**
     * Get the defined statuses for a product.
     *
     * @return array
     */
    public static function getStatuses(): array
    {
        return [
            'draft' => 'Draft',
            'published' => 'Published',
            'archived' => 'Archived',
            'out_of_stock' => 'Out of Stock',
            // 'pending_review' => 'Pending Review', // If admin approval is needed
            // 'unavailable' => 'Unavailable', // More permanent than out_of_stock
        ];
    }

    /**
     * Scope a query to only include published products.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopePublished($query)
    {
        return $query->where('status', self::STATUS_PUBLISHED)->where('is_published', true);
    }
}
