<?php

namespace Modules\BusinessDirectory\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Modules\BusinessDirectory\Entities\Product; // Explicit import for Product

class OrderItem extends Model
{
    use HasFactory;

    protected $table = 'bd_order_items';

    public $timestamps = true; // Ensure timestamps are managed

    protected $fillable = [
        'order_id',
        'product_id',
        'quantity',
        'price_per_unit', // Price of the product at the time of order
        'total_price',
    ];

    protected $casts = [
        'quantity' => 'integer',
        'price_per_unit' => 'decimal:2',
        'total_price' => 'decimal:2',
    ];

    /**
     * The order this item belongs to.
     */
    public function order(): BelongsTo
    {
        return $this->belongsTo(Order::class);
    }

    /**
     * The product associated with this order item.
     */
    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    /**
     * Calculate total price before saving if not provided.
     */
    public function setTotalPriceAttribute($value)
    {
        $this->attributes['total_price'] = $value ?? ($this->attributes['quantity'] * $this->attributes['price_per_unit']);
    }

    protected static function newFactory() {
        return \Modules\BusinessDirectory\Database\factories\OrderItemFactory::new();
    }
}
