<?php

namespace Modules\BusinessDirectory\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;
use App\Models\Team; // Import Jetstream Team model

class CompanyProject extends Model
{
    use HasFactory, HasSlug;

    protected $table = 'bd_projects';

    protected $fillable = [
        'company_id',
        'name',
        'user_id',
        'slug',
        'description',
        'start_date',
        'end_date',
        'budget',
        'currency',
        'status', // e.g., draft, planning, in_progress, completed, on_hold, cancelled
        'client_company_id',
        'team_id', // Add team_id
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'budget' => 'decimal:2',
    ];

    /**
     * Get the options for generating the slug.
     */
    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()
            ->generateSlugsFrom('name')
            ->saveSlugsTo('slug');
    }

    /**
     * Get the company that owns the project.
     */
    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class, 'company_id');
    }

    /**
     * Get the client company for this project (if applicable).
     */
    public function clientCompany(): BelongsTo
    {
        return $this->belongsTo(Company::class, 'client_company_id');
    }

    /**
     * Get the tasks associated with the project.
     */
    public function tasks(): HasMany
    {
        return $this->hasMany(CompanyTask::class, 'project_id');
    }

    /**
     * Get the team associated with the project.
     */
    public function team(): BelongsTo
    {
        return $this->belongsTo(Team::class, 'team_id');
    }

    /**
     * Factory
     */
    protected static function newFactory()
    {
        return \Modules\BusinessDirectory\Database\factories\CompanyProjectFactory::new();
    }

    /**
     * Get the defined statuses for a project.
     *
     * @return array
     */
    public static function getStatuses(): array
    {
        return [
            'draft' => 'Draft',
            'planning' => 'Planning',
            'in_progress' => 'In Progress',
            'completed' => 'Completed',
            'on_hold' => 'On Hold',
            'cancelled' => 'Cancelled',
        ];
    }
}
