<?php

namespace Modules\AppManager\Http\Controllers\Api;

use App\Http\Controllers\Controller; // Assuming your base controller is in App\Http\Controllers
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Modules\AppManager\Models\License; // Assuming you have a License model
use Modules\AppManager\Models\ManagedScript; // Assuming you have a ManagedScript model for products

class ActivationValidationController extends Controller
{
    public function validateActivation(Request $request)
    {
        $request->validate([
            'activation_token' => 'required|string',
            'domain' => 'required|string|url',
            'product_slug' => 'required|string',
        ]);

        $activationToken = $request->input('activation_token');
        $domain = strtolower(trim($request->input('domain')));
        $productSlug = $request->input('product_slug');

        try {
            $license = License::where('activation_token', $activationToken)->first();

            if (!$license) {
                return response()->json([
                    'status' => 'error',
                    'is_valid' => false,
                    'message' => 'Activation token not found.',
                ], 404);
            }

            // Assuming License model has a relationship to ManagedScript (product)
            // and ManagedScript has a 'slug' field.
            if (!$license->managedScript || $license->managedScript->slug !== $productSlug) {
                return response()->json([
                    'status' => 'error',
                    'is_valid' => false,
                    'message' => 'Product slug mismatch for this token.',
                ], 400);
            }

            // Ensure domain is consistently stored/checked in lowercase
            if (strtolower(trim($license->activated_domain)) !== $domain) {
                return response()->json([
                    'status' => 'error',
                    'is_valid' => false,
                    'message' => 'Domain mismatch for this token.',
                ], 400);
            }

            // Check license status (e.g., 'active', 'expired', 'revoked')
            // This logic depends on how you manage license statuses in your AppManager
            if ($license->status !== 'active') { // Adjust 'active' to your actual active status value
                return response()->json([
                    'status' => 'error',
                    'is_valid' => false,
                    'message' => "License is not active. Current status: {$license->status}.",
                ], 400);
            }

            // If all checks pass
            Log::info('AppManager: Activation token validated successfully.', ['token' => $activationToken, 'domain' => $domain, 'product_slug' => $productSlug]);
            return response()->json([
                'status' => 'success',
                'is_valid' => true,
                'message' => 'Token is valid.',
            ]);

        } catch (\Exception $e) {
            Log::error('AppManager: Error during activation validation API call.', ['error' => $e->getMessage(), 'token' => $activationToken, 'domain' => $domain]);
            return response()->json(['status' => 'error', 'is_valid' => false, 'message' => 'An internal error occurred during validation.'], 500);
        }
    }
}