<?php

namespace Modules\WebPilotAI\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt; // Import Crypt facade
use Illuminate\Validation\Rule; // Import Rule
use Modules\WebPilotAI\Models\AIModel;
use Illuminate\Support\Facades\Storage;
use Modules\WebPilotAI\Models\Website;
use Modules\WebPilotAI\Jobs\ProcessWebsiteGeneration; // Import the Job
use Modules\WebPilotAI\Jobs\DeployWebsiteToHosting; // Placeholder for new Job

class WebsiteController extends Controller
{
    /**
     * Display a listing of the user's websites.
     */
    public function index()
    {
        $websites = Website::where('user_id', Auth::id())
                            ->with('aiModel') // Eager load the AI model relationship
                            ->orderBy('created_at', 'desc')
                            ->paginate(10); // Paginate results

        return view('webpilotai::user.websites.index', compact('websites'));
    }

    /**
     * Show the form for creating a new website.
     */
    public function create()
    {
        // Fetch active AI models for the user to choose from
        $aiModels = AIModel::where('is_active', true)->orderBy('name')->get();
        return view('webpilotai::user.websites.create', compact('aiModels'));
    }

    /**
     * Store a newly created website in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $validatedData = $request->validate([
            'name' => 'nullable|string|max:255',
            'description_prompt' => 'required|string|min:20',
            'ai_model_id' => [
                'required',
                Rule::exists('ai_models', 'id')->where(function ($query) {
                    $query->where('is_active', true);
                }),
            ],
        ]);

        $website = Website::create([
            'user_id' => Auth::id(),
            'name' => $validatedData['name'] ?? 'My AI Website - ' . now()->format('Y-m-d H:i'),
            'description_prompt' => $validatedData['description_prompt'],
            'ai_model_id' => $validatedData['ai_model_id'],
            'status' => 'pending_generation', // Initial status
        ]);

        // Dispatch the job to handle AI generation
        ProcessWebsiteGeneration::dispatch($website);

        return redirect()->route('frontend.webpilotai.websites.index')->with('success', 'Your website request has been submitted! Generation will begin shortly.');
    }

    /**
     * Display the specified website.
     */
    public function show(Website $website) // Route model binding
    {
        // Ensure the authenticated user owns this website
        $this->authorize('view', $website); // Requires a WebsitePolicy

        return view('webpilotai::user.websites.show', compact('website'));
    }

    /**
     * Show the form for editing the specified website.
     */
    public function edit(Website $website) // Route model binding
    {
        $this->authorize('update', $website); // Requires a WebsitePolicy
        $aiModels = AIModel::where('is_active', true)->orderBy('name')->get();

        return view('webpilotai::user.websites.edit', compact('website', 'aiModels'));
    }

    /**
     * Update the specified website in storage.
     */
    public function update(Request $request, Website $website): RedirectResponse
    {
        $this->authorize('update', $website);

        $validatedData = $request->validate([
            'name' => 'nullable|string|max:255',
            'description_prompt' => 'required|string|min:20',
            'ai_model_id' => [
                'required',
                Rule::exists('ai_models', 'id')->where(function ($query) {
                    $query->where('is_active', true);
                }),
            ],
            'regenerate' => 'nullable|boolean', // Checkbox to trigger regeneration
        ]);

        $originalPrompt = $website->description_prompt;
        $originalModelId = $website->ai_model_id;

        $website->update([
            'name' => $validatedData['name'] ?? $website->name,
            'description_prompt' => $validatedData['description_prompt'],
            'ai_model_id' => $validatedData['ai_model_id'],
        ]);

        // Re-dispatch job if prompt changed, model changed, or user explicitly requests regeneration
        if ($request->has('regenerate') || $validatedData['description_prompt'] !== $originalPrompt || $validatedData['ai_model_id'] != $originalModelId) {
            $website->update(['status' => 'pending_regeneration', 'generated_content_path' => null, 'generation_error' => null]); // Reset path and error
            ProcessWebsiteGeneration::dispatch($website);
            return redirect()->route('frontend.webpilotai.websites.show', $website->id)->with('success', 'Website details updated. Regeneration has started!');
        }

        return redirect()->route('frontend.webpilotai.websites.show', $website->id)->with('success', 'Website details updated successfully.');
    }

    /**
     * Remove the specified website from storage.
     */
    public function destroy(Website $website): RedirectResponse
    {
        $this->authorize('delete', $website);

        // Delete the generated file/directory from storage
        if ($website->generated_content_path) {
            // The path stored is relative to public/storage, e.g., "storage/webpilotai_sites_zip/1/file.zip"
            // To use Storage facade, we need path relative to disk's root, e.g., "public/webpilotai_sites_zip/1/file.zip" for the 'public' disk
            $storagePath = str_replace('storage/', 'public/', $website->generated_content_path);
            
            if (Storage::exists($storagePath)) {
                Storage::delete($storagePath); // Deletes the ZIP file
                
                // Attempt to delete the parent directory if it's empty
                $directory = dirname($storagePath);
                if (empty(Storage::files($directory)) && empty(Storage::directories($directory))) {
                    Storage::deleteDirectory($directory);
                }
            }
        }
        $website->delete();
        return redirect()->route('frontend.webpilotai.websites.index')->with('success', 'Website deleted successfully.');
    }

    /**
     * Show the form for configuring deployment for a specific website.
     */
    public function showDeployForm(Website $website)
    {
        $this->authorize('update', $website); // Users can only deploy sites they own/can update

        if ($website->status !== 'completed' || !$website->generated_content_path) {
            return redirect()->route('frontend.webpilotai.websites.show', $website->id)
                             ->withErrors(['msg' => 'Website must be successfully generated before it can be deployed.']);
        }

        // Retrieve last used deployment details if available (e.g., from $website->deployment_details)
        // For now, we'll pass an empty array or defaults.
        $lastDeploymentDetails = $website->deployment_details ?? [];

        return view('webpilotai::user.websites.deploy', compact('website', 'lastDeploymentDetails'));
    }

    /**
     * Handle the submission of deployment details and initiate deployment.
     */
    public function handleDeployment(Request $request, Website $website): RedirectResponse
    {
        $this->authorize('update', $website);

        $validatedData = $request->validate([
            'deployment_type' => ['required', Rule::in(['ftp', 'sftp', 'cpanel'])],
            'host' => 'required|string|max:255',
            'username' => 'required|string|max:255',
            'password' => 'required|string', // Consider not validating length too strictly here
            'port' => 'nullable|numeric',
            'remote_path' => 'nullable|string|max:255', // e.g., public_html/my-site
            'cpanel_domain' => 'nullable|required_if:deployment_type,cpanel|string|max:255',
        ]);

        // Encrypt the password before dispatching the job
        $deploymentPayload = $validatedData;
        if (isset($deploymentPayload['password'])) {
            $deploymentPayload['password'] = Crypt::encryptString($deploymentPayload['password']);
        }

        DeployWebsiteToHosting::dispatch($website, $deploymentPayload);

        return redirect()->route('frontend.webpilotai.websites.show', $website->id)->with('success', 'Deployment process initiated. You will be notified upon completion.');
    }

    /**
     * Get the current status of a website for AJAX polling.
     */
    public function getDeploymentStatus(Website $website): JsonResponse
    {
        $this->authorize('view', $website);

        return response()->json([
            'name' => $website->name ?: 'Untitled Website',
            'status' => $website->status,
            'status_friendly' => Str::title(str_replace('_', ' ', $website->status)),
            'generation_error' => $website->generation_error,
            'deployment_details' => $website->deployment_details,
            'last_generated_at' => $website->last_generated_at ? $website->last_generated_at->format('M d, Y H:i A') : 'Never',
            'generated_content_path' => $website->generated_content_path ? asset($website->generated_content_path) : null,
            'can_download' => $website->status === Website::STATUS_COMPLETED && $website->generated_content_path,
            'can_deploy_form' => $website->status === Website::STATUS_COMPLETED && $website->generated_content_path,
        ]);
    }
}
