<?php

namespace Modules\WebPilotAI\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Http; // Import Http facade
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Modules\WebPilotAI\Models\Website;
use App\Models\Setting; // Import Setting model
use Modules\WebPilotAI\Models\AITemplate; // Import AITemplate model
use Modules\WebPilotAI\Models\AIStylePreset; // Import AIStylePreset model
use Modules\WebPilotAI\Events\WebsiteGenerationSucceeded; // Import Success Event
use Modules\WebPilotAI\Events\WebsiteGenerationFailed; // Import Failure Event
use Google\Cloud\AIPlatform\V1\EndpointServiceClient; // For Google AI (if using Vertex AI)
use Throwable;
use ZipArchive; // For zipping

class ProcessWebsiteGeneration implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public Website $website;

    /**
     * Create a new job instance.
     */
    public function __construct(Website $website)
    {
        $this->website = $website;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        Log::info("[ProcessWebsiteGeneration@handle] Job picked up for website ID: {$this->website->id}. Initial status from job's perspective: {$this->website->status}");
        Log::info("Starting AI website generation for website ID: {$this->website->id}");
        $this->website->update(['status' => 'generating']);

        $tempSiteDir = null; // Initialize temp directory variable

        try {
            // Eager load aiModel, aiTemplate, aiStylePresets, and assets relationships
            $website = $this->website->load(['user', 'aiModel', 'aiTemplate', 'aiStylePresets', 'assets']);
            $aiModel = $website->aiModel;

            if (!$aiModel || !$aiModel->identifier) {
                throw new \Exception("AI Model not found, not associated, or identifier missing for website ID: {$this->website->id}");
            }

            $apiKey = null;
            if ($aiModel->api_key_setting_name) {
                // Fetch API key from the database settings table
                $settingKey = $aiModel->api_key_setting_name; // The full key is already stored
                $apiKey = Setting::getValue($settingKey); // Use the static getValue method
                Log::debug("ProcessWebsiteGeneration: Fetched API Key setting key: " . $settingKey);
                if ($apiKey) {
                   Log::debug("ProcessWebsiteGeneration: API Key found (encrypted) - length: " . strlen($apiKey));
                } else Log::warning("ProcessWebsiteGeneration: API Key NOT FOUND for key: " . $settingKey);

            }

            if (($aiModel->provider === 'OpenAI' || $aiModel->provider === 'Anthropic' || $aiModel->provider === 'GoogleAI') && !$apiKey) {
                 throw new \Exception("API key for model '{$aiModel->name}' (setting: '{$aiModel->api_key_setting_name}') is not configured in module settings.");
            }


            // Construct a detailed prompt for the AI
            $userPrompt = $this->website->description_prompt;
            $websiteName = $this->website->name ?: "AI Generated Website";

            // --- Updated System Prompt for Multi-page and Assets ---
            $systemPrompt = "You are an expert web developer AI. Your task is to generate a complete, responsive, and visually appealing website with multiple pages and necessary assets.\n";
            $systemPrompt .= "Generate the content for each page as separate HTML files (e.g., index.html, about.html, contact.html).\n";
            $systemPrompt .= "Generate a single CSS file (style.css) for all styles and a single JavaScript file (script.js) for basic interactivity.\n";
            $systemPrompt .= "Link the CSS and JS files correctly in each HTML page's &lt;head&gt; and before &lt;/body&gt; respectively, assuming they will be in an 'assets' subdirectory (e.g., &lt;link rel=\"stylesheet\" href=\"assets/css/style.css\"&gt;, &lt;script src=\"assets/js/script.js\"&gt;&lt;/script&gt;).\n";
            $systemPrompt .= "If specific images are mentioned or provided (you will be told about them), use relative paths assuming they will be in 'assets/images' (e.g., &lt;img src=\"assets/images/your_image.jpg\"&gt;).\n";
            $systemPrompt .= "Provide the content for each file clearly separated, using markdown headers like `## File: index.html`, `## File: assets/css/style.css`, `## File: assets/js/script.js`, etc.\n";
            // Instruct AI on how to reference provided assets
            $systemPrompt .= "When using provided images, reference them using the relative path 'assets/images/filename.ext'. For example, if an image 'hero.jpg' is provided, use &lt;img src=\"assets/images/hero.jpg\"&gt;.\n";
            $systemPrompt .= "Do NOT include any other explanations or markdown formatting outside these file blocks.\n";
            $systemPrompt .= "Ensure the output for each file is ONLY the raw code for that file.";
            // --- End Updated System Prompt ---

            // --- Build Dynamic Prompt Components ---

            // 1. Pages List
            $pagesListString = "";
            $pagesToGenerateString = "Based on the description, generate the following pages:\n";
            if (!empty($website->pages_structure)) {
                foreach ($website->pages_structure as $page) {
                    $pagesToGenerateString .= "- {$page['name']} (filename: {$page['filename']})\n";
                }
            } else { // Fallback if pages_structure is somehow empty, though controller should provide defaults
                $pagesToGenerateString .= "- Home (filename: index.html)\n- About Us (filename: about.html)\n- Contact (filename: contact.html)\n";
            }
            $pagesListString = $pagesToGenerateString . "\nAlso generate a style.css and a script.js file to be placed in 'assets/css/' and 'assets/js/' respectively.\n";

            // 2. Assets List
            $assetsForPromptString = "";
            $usedTargetFilenames = [];

            if ($website->assets->isNotEmpty()) {
                foreach ($website->assets as $asset) {
                    $originalName = $asset->original_name;
                    $targetFilename = $originalName;
                    $counter = 1;
                    $nameParts = pathinfo($originalName);
                    $baseName = $nameParts['filename'];
                    $extension = isset($nameParts['extension']) ? '.' . $nameParts['extension'] : '';

                    // Ensure unique target filename within the assets/images directory
                    while (in_array($targetFilename, $usedTargetFilenames)) {
                        $targetFilename = $baseName . '-' . $counter++ . $extension;
                    }
                    $usedTargetFilenames[] = $targetFilename;
                    $asset->target_filename_for_zip = $targetFilename; // Store for later use during copy

                    // Include description if available
                    $description = $asset->description ? ", Description: '{$asset->description}'" : '';

                    $assetDetailsForPrompt[] = "- Original name: '{$originalName}', Use in HTML as: 'assets/images/{$targetFilename}' (Type: {$asset->type}){$description}";
                }
            }

            if (!empty($assetDetailsForPrompt)) { // Use $assetDetailsForPrompt built above
                $assetsForPromptString = "\n\n--- AVAILABLE ASSETS START ---\n";
                $assetsForPromptString .= "The following assets have been provided. When using an image in an `<img>` tag, use the 'Use in HTML as' path provided for each asset:\n" . implode("\n", $assetDetailsForPrompt);
                $assetsForPromptString .= "\n--- AVAILABLE ASSETS END ---";
            }

            // 3. Styling Guidelines
            $stylingForPromptString = "";
            $aiTemplate = $this->website->aiTemplate;

            $styleContext = [
                'colors' => [], // e.g., 'primary' => '#RRGGBB', 'background' => '#RRGGBB'
                'fonts' => [],  // e.g., 'headings' => 'Arial, sans-serif', 'body' => 'Roboto, sans-serif'
                'theme_keywords' => [],
                'layout_hints' => [], // e.g., 'spacing' => 'generous', 'corners' => 'rounded'
                'element_specific' => [], // e.g., 'buttons' => ['shape' => 'pill', 'hover' => 'shadow']
                'raw_instructions' => [], // For direct pass-through instructions
            ];

            if ($this->website->aiStylePresets->isNotEmpty()) {
                Log::info("Website ID {$this->website->id}: Applying AI Style Presets.");
                foreach ($this->website->aiStylePresets as $stylePreset) {
                    Log::info("Website ID {$this->website->id}: Processing Style Preset '{$stylePreset->name}' (Type: {$stylePreset->type}). Configuration: " . json_encode($stylePreset->configuration));
                    if ($stylePreset->configuration && (is_object($stylePreset->configuration) || is_array($stylePreset->configuration))) {
                        $config = (array) $stylePreset->configuration; // Cast to array for easier handling

                        switch ($stylePreset->type) {
                            case 'color_palette':
                                foreach (['primary', 'secondary', 'accent', 'background', 'text', 'text_on_primary', 'text_on_secondary'] as $colorKey) {
                                    if (isset($config[$colorKey])) {
                                        if (is_string($config[$colorKey])) {
                                            $styleContext['colors'][$colorKey] = $config[$colorKey];
                                        } elseif (is_array($config[$colorKey]) && isset($config[$colorKey]['value'])) {
                                            // Handle more complex color definitions if they have a 'value'
                                            $styleContext['colors'][$colorKey] = $config[$colorKey]['value'];
                                            if (isset($config[$colorKey]['description'])) {
                                                $styleContext['raw_instructions'][] = "The {$colorKey} color ({$config[$colorKey]['value']}) is intended for: {$config[$colorKey]['description']}.";
                                            }
                                        }
                                    }
                                }
                                break;
                            case 'font_scheme':
                                foreach (['headings', 'body', 'captions', 'buttons'] as $fontTarget) {
                                    if (isset($config[$fontTarget])) {
                                        if (is_string($config[$fontTarget])) { // Simple font family string
                                            $styleContext['fonts'][$fontTarget] = $config[$fontTarget];
                                        } elseif (is_array($config[$fontTarget]) && isset($config[$fontTarget]['family'])) { // Complex font object
                                            $fontDetails = "font-family: '{$config[$fontTarget]['family']}'";
                                            if (isset($config[$fontTarget]['weight'])) $fontDetails .= "; font-weight: {$config[$fontTarget]['weight']}";
                                            if (isset($config[$fontTarget]['size'])) $fontDetails .= "; font-size: {$config[$fontTarget]['size']}";
                                            if (isset($config[$fontTarget]['style'])) $fontDetails .= "; font-style: {$config[$fontTarget]['style']}";
                                            $styleContext['fonts'][$fontTarget] = $fontDetails;
                                        }
                                    }
                                }
                                break;
                            case 'visual_theme':
                                if (isset($config['keywords']) && is_array($config['keywords'])) {
                                    $styleContext['theme_keywords'] = array_merge($styleContext['theme_keywords'], $config['keywords']);
                                }
                                if (isset($config['mood']) && is_string($config['mood'])) {
                                    $styleContext['raw_instructions'][] = "The overall mood should be: {$config['mood']}.";
                                }
                                foreach (['spacing', 'corner_style', 'shadow_intensity'] as $layoutKey) {
                                    if (isset($config[$layoutKey]) && is_string($config[$layoutKey])) {
                                        $styleContext['layout_hints'][$layoutKey] = $config[$layoutKey];
                                    }
                                }
                                break;
                            case 'element_specific_styles': // New type for granular control
                                if (is_array($config)) {
                                    foreach ($config as $elementKey => $elementStyles) {
                                        $styleContext['element_specific'][$elementKey] = array_merge($styleContext['element_specific'][$elementKey] ?? [], (array)$elementStyles);
                                    }
                                }
                                break;
                            case 'direct_instruction': // For presets that just contain raw instructions
                                if (isset($config['instruction']) && is_string($config['instruction'])) {
                                    $styleContext['raw_instructions'][] = $config['instruction'];
                                } elseif (isset($config['instructions']) && is_array($config['instructions'])) {
                                    $styleContext['raw_instructions'] = array_merge($styleContext['raw_instructions'], $config['instructions']);
                                }
                                break;
                        }
                    }
                }
            }

            // Consolidate unique keywords
            $styleContext['theme_keywords'] = array_unique($styleContext['theme_keywords']);
            $styleContext['raw_instructions'] = array_unique($styleContext['raw_instructions']);

            // Build the final style instructions for the prompt
            $structuredStyleInstructions = [];

            if (!empty($styleContext['colors'])) {
                $colorDetails = [];
                foreach ($styleContext['colors'] as $key => $value) {
                    $colorDetails[] = "  - " . Str::title(str_replace('_', ' ', $key)) . ": {$value}";
                }
                $structuredStyleInstructions[] = "Colors:\n" . implode("\n", $colorDetails);
            }

            if (!empty($styleContext['fonts'])) {
                $fontDetails = [];
                foreach ($styleContext['fonts'] as $key => $value) {
                    $fontDetails[] = "  - " . Str::title(str_replace('_', ' ', $key)) . ": '{$value}'";
                }
                 $structuredStyleInstructions[] = "Fonts:\n" . implode("\n", $fontDetails);
            }

            if (!empty($styleContext['theme_keywords'])) {
                $structuredStyleInstructions[] = "Theme Keywords: [" . implode(', ', $styleContext['theme_keywords']) . "]";
            }

            if (!empty($styleContext['layout_hints'])) {
                $layoutDetails = [];
                foreach ($styleContext['layout_hints'] as $key => $value) {
                    $layoutDetails[] = "  - " . Str::title(str_replace('_', ' ', $key)) . ": {$value}";
                }
                $structuredStyleInstructions[] = "Layout Hints:\n" . implode("\n", $layoutDetails);
            }

            if (!empty($styleContext['element_specific'])) {
                $elementDetails = [];
                foreach ($styleContext['element_specific'] as $element => $styles) {
                    $elementStyleDetails = [];
                    foreach ($styles as $prop => $val) {
                        $elementStyleDetails[] = "    - " . Str::title(str_replace('_', ' ', $prop)) . ": {$val}";
                    }
                    if ($elementStyleDetails) {
                         $elementDetails[] = "  - " . Str::title(str_replace('_', ' ', $element)) . ":\n" . implode("\n", $elementStyleDetails);
                    }
                }
                if ($elementDetails) $structuredStyleInstructions[] = "Element Specific Styles:\n" . implode("\n", $elementDetails);
            }

            if (!empty($styleContext['raw_instructions'])) {
                $structuredStyleInstructions[] = "Raw Instructions:\n- " . implode("\n- ", $styleContext['raw_instructions']);
            }

            if (!empty($structuredStyleInstructions)) {
                $stylingForPromptString = "\n\n--- STYLING GUIDELINES START ---\n" . implode("\n\n", $structuredStyleInstructions) . "\n--- STYLING GUIDELINES END ---";
            }

            // --- Assemble Final Prompt based on Template ---

            $baseUserPrompt = "Generate an HTML page for a website named '{$websiteName}'.\nUser Description: \"{$userPrompt}\"\n\n";
            $finalConstructedPrompt = ""; // Initialize

            if ($aiTemplate && !empty($aiTemplate->prompt_modifications)) {
                Log::info("Website ID {$this->website->id}: Using AI Template '{$aiTemplate->name}' (ID: {$aiTemplate->id}). Modifications: " . json_encode($aiTemplate->prompt_modifications));
            }

            if ($aiTemplate && isset($aiTemplate->prompt_modifications['full_prompt_template']) && is_string($aiTemplate->prompt_modifications['full_prompt_template']) && !empty(trim($aiTemplate->prompt_modifications['full_prompt_template']))) {
                // Use the full template and replace placeholders
                $templateContent = $aiTemplate->prompt_modifications['full_prompt_template'];
                $finalConstructedPrompt = str_replace(
                    ['[WEBSITE_NAME]', '[USER_PROMPT]', '[PAGES_LIST]', '[ASSETS_LIST]', '[STYLING_GUIDELINES]'],
                    // Ensure $userPrompt is used here instead of $baseUserPrompt for the [USER_PROMPT] placeholder
                    [$websiteName, $userPrompt, $pagesListString, $assetsForPromptString, $stylingForPromptString],
                    $templateContent
                );
            } else {
                // No full template, append dynamic sections to base prompt
                $finalConstructedPrompt .= $pagesListString . "Ensure the design is coherent and professional. Use appropriate HTML5 semantic tags.";
                $finalConstructedPrompt .= $assetsForPromptString;
                $finalConstructedPrompt .= $stylingForPromptString;
                // Prepend the base user prompt to the dynamically assembled parts
                $finalConstructedPrompt = $baseUserPrompt . $finalConstructedPrompt;

                // Apply prefix/suffix if they exist and no full template was used
                if ($aiTemplate && !empty($aiTemplate->prompt_modifications)) {
                     $structure = $aiTemplate->prompt_modifications;
                     if (isset($structure['prefix']) && is_string($structure['prefix'])) {
                        $finalConstructedPrompt = rtrim($structure['prefix']) . "\n\n" . ltrim($finalConstructedPrompt);
                    }
                    if (isset($structure['suffix']) && is_string($structure['suffix'])) {
                        $finalConstructedPrompt = rtrim($finalConstructedPrompt) . "\n\n" . ltrim($structure['suffix']);
                    }
                }
            }

            Log::debug("Final AI Prompt for website ID {$this->website->id}: {$finalConstructedPrompt}");

            $generatedContent = ""; // This will hold the raw response from the AI

            if ($aiModel->provider === 'OpenAI') {
                $client = \OpenAI::client($apiKey);
                $response = $client->chat()->create([
                    'model' => $aiModel->identifier,
                    'messages' => [
                        ['role' => 'system', 'content' => $systemPrompt],
                        ['role' => 'user', 'content' => $finalConstructedPrompt], // Use the final constructed prompt
                    ],
                    'temperature' => 0.7, // Adjust for creativity
                    'max_tokens' => 4000, // Increased max_tokens for potentially larger output
                ]);
                $generatedContent = $response->choices[0]->message->content;

            } elseif ($aiModel->provider === 'Anthropic') {
                if (!$apiKey) {
                    throw new \Exception("Anthropic API key for model '{$aiModel->name}' (setting: '{$aiModel->api_key_setting_name}') is not configured in module settings.");
                }
                Log::info("Anthropic provider selected for website ID: {$this->website->id}. Attempting API call.");

                // Anthropic API endpoint for messages
                $anthropicApiUrl = 'https://api.anthropic.com/v1/messages';

                // Note: Anthropic's API is slightly different. It doesn't use a 'system' role in the same way.
                // We can prepend system-like instructions to the user prompt or use the 'system' parameter if available for the model.
                // For Messages API, the 'system' parameter is preferred.
                $response = Http::withHeaders([
                    'x-api-key' => $apiKey,
                    'anthropic-version' => '20233-10-01', // Use the latest version that supports the 'system' parameter
                    'content-type' => 'application/json',
                ])->timeout(180)->post($anthropicApiUrl, [ // Increased timeout for potentially longer generations
                    'model' => $aiModel->identifier,
                    'system' => $systemPrompt, // Pass the system prompt here
                    'messages' => [
                        ['role' => 'user', 'content' => $finalConstructedPrompt], // Use the final constructed prompt
                    ],
                    'max_tokens' => 4000, // Adjust as needed, Claude models can handle large contexts
                    'temperature' => 0.7,
                ]);

                if ($response->failed()) {
                    Log::error("Anthropic API call failed for website ID {$this->website->id}: " . $response->body());
                    throw new \Exception("Anthropic API call failed: " . $response->status() . " - " . $response->body());
                }

                $responseData = $response->json();
                // The main content is usually in responseData['content'][0]['text']
                $generatedContent = $responseData['content'][0]['text'] ?? '';

            } elseif ($aiModel->provider === 'GoogleAI') {
                 if (!$apiKey) {
                    throw new \Exception("Google AI API key for model '{$aiModel->name}' (setting: '{$aiModel->api_key_setting_name}') is not configured in module settings.");
                }
                Log::info("GoogleAI provider selected for website ID: {$this->website->id}. Attempting API call.");

                // Using the google/generative-ai package
                // Ensure you have 'gemini-pro' or a similar model identifier in your AIModel record.
                // The package handles API key authentication via environment variables (GOOGLE_API_KEY)
                // or by explicitly passing it. Since we fetch it from settings, we'll use it directly.

                try {
                    $client = \Google\AI\GenerativeAI::client($apiKey);
                    $model = $client->geminiPro(); // Or specific model like gemini-1.5-pro-latest if $aiModel->identifier is just 'gemini-pro'
                                                  // If $aiModel->identifier is 'models/gemini-1.5-pro-latest', then use:
                                                  // $model = $client->generativeModel($aiModel->identifier);

                    // Constructing the content for Google AI
                    // Google's API often uses a sequence of 'user' and 'model' turns.
                    // System prompts can be part of the initial user message or a separate 'system_instruction' if supported by the specific model version/API.
                    // For Gemini, system instructions are typically set at the model level or as part of the first message.
                    // We'll combine system and user prompt for simplicity here.
                    $combinedPrompt = $systemPrompt . "\n\n" . $finalConstructedPrompt;

                    $response = $model->generateContent($combinedPrompt);
                    $generatedContent = $response->text();

                } catch (\Exception $e) {
                    Log::error("Google AI API call failed for website ID {$this->website->id}: " . $e->getMessage());
                    throw new \Exception("Google AI API call failed: " . $e->getMessage());
                }

            } else {
                // Placeholder for other AI providers or if no provider matches
                Log::warning("AI Provider '{$aiModel->provider}' not implemented or no API key for website ID: {$this->website->id}. Using placeholder content.");
                // Mimic the expected multi-file output format
                $generatedContent = "## File: index.html\n&lt;!DOCTYPE html&gt;&lt;html&gt;&lt;head&gt;&lt;title&gt;".$websiteName." (Placeholder)&lt;/title&gt;&lt;link rel=\"stylesheet\" href=\"assets/css/style.css\"&gt;&lt;/head&gt;&lt;body&gt;&lt;h1&gt;Welcome to ".$websiteName."&lt;/h1&gt;&lt;p&gt;AI Provider '{$aiModel->provider}' not fully implemented yet.&lt;/p&gt;&lt;p&gt;Original Prompt: {$userPrompt}&lt;/p&gt;&lt;a href=\"about.html\"&gt;About&lt;/a&gt;&lt;script src=\"assets/js/script.js\"&gt;&lt;/script&gt;&lt;/body&gt;&lt;/html&gt;\n\n";
                $generatedContent .= "## File: about.html\n&lt;!DOCTYPE html&gt;&lt;html&gt;&lt;head&gt;&lt;title&gt;About Us - ".$websiteName."&lt;/title&gt;&lt;link rel=\"stylesheet\" href=\"assets/css/style.css\"&gt;&lt;/head&gt;&lt;body&gt;&lt;h1&gt;About Us&lt;/h1&gt;&lt;p&gt;This is the about page.&lt;/p&gt;&lt;a href=\"index.html\"&gt;Home&lt;/a&gt;&lt;script src=\"assets/js/script.js\"&gt;&lt;/script&gt;&lt;/body&gt;&lt;/html&gt;\n\n";
                $generatedContent .= "## File: assets/css/style.css\nbody { font-family: sans-serif; margin: 20px; background-color: #f0f0f0; }\nh1 { color: #333; }\n\n";
                $generatedContent .= "## File: assets/js/script.js\nconsole.log('Website loaded!');\n";
            }

            if (empty($generatedContent)) {
                throw new \Exception("AI returned empty content for website ID: {$this->website->id}");
            }

            // --- New: Parse AI response into files and save to temp directory ---
            $tempSiteDir = storage_path('app/temp_site_' . $this->website->id . '_' . time());
            File::ensureDirectoryExists($tempSiteDir, 0755, true);
            Log::info("Created temporary site directory: {$tempSiteDir} for website ID: {$this->website->id}");

            // Simple parsing based on markdown headers (## File: filename)
            // Regex to capture filename and content, handling potential markdown code blocks
            preg_match_all('/## File: (.*?)\n(?:```[a-z]*\n)?(.*?)(?:\n```)?(?=\n## File:|\z)/s', $generatedContent, $matches, PREG_SET_ORDER);

            $savedFilesCount = 0;
            if (empty($matches) && Str::contains($generatedContent, '## File:')) { // Fallback for simpler parsing if regex fails but marker exists
                $fileBlocks = explode('## File: ', $generatedContent);
                array_shift($fileBlocks); // Remove content before the first "## File:"
                foreach ($fileBlocks as $block) {
                    if (trim($block) === '') continue;
                    list($filename, $content) = explode("\n", $block, 2);
                    $filename = trim($filename);
                    $content = trim($content);
                     // Remove potential markdown code block fences if AI adds them
                    if (Str::startsWith($content, '```') && Str::endsWith($content, '```')) {
                        $content = preg_replace('/^```[a-z]*\n/', '', $content); // Remove opening fence
                        $content = preg_replace('/\n```$/', '', $content);        // Remove closing fence
                    }
                    $matches[] = [1 => $filename, 2 => $content]; // Add to matches array for unified processing
                }
            }

            foreach ($matches as $match) {
                $filename = trim($match[1]);
                $content = trim($match[2]);

                if (empty($filename) || empty($content)) {
                    Log::warning("Skipping empty file block or missing filename for website ID {$this->website->id}. Filename: '{$filename}'");
                    continue;
                }

                $filePath = $tempSiteDir . DIRECTORY_SEPARATOR . str_replace('/', DIRECTORY_SEPARATOR, $filename); // Ensure correct directory separators
                File::ensureDirectoryExists(dirname($filePath));
                File::put($filePath, $content);
                Log::debug("Saved file: {$filename} to {$filePath} for website ID: {$this->website->id}");
                $savedFilesCount++;
            }

            if ($savedFilesCount === 0) {
                 throw new \Exception("AI response parsed, but no files were saved for website ID: {$this->website->id}. Check AI output format. Raw AI output: " . Str::limit($generatedContent, 500));
            }
            Log::info("Successfully parsed and saved {$savedFilesCount} files to temporary directory {$tempSiteDir} for website ID: {$this->website->id}");

            // --- New: Copy uploaded assets into the temporary site directory ---
            if ($website->assets->isNotEmpty()) {
                $targetAssetsDir = $tempSiteDir . DIRECTORY_SEPARATOR . 'assets' . DIRECTORY_SEPARATOR . 'images';
                File::ensureDirectoryExists($targetAssetsDir, 0755, true);
                Log::info("Ensured target assets directory exists: {$targetAssetsDir}");

                foreach ($website->assets as $asset) {
                    $sourcePath = Storage::disk($asset->disk)->path($asset->path);
                    // Use the pre-determined unique target filename
                    $targetFilenameInZip = $asset->target_filename_for_zip ?? $asset->original_name;
                    $targetPath = $targetAssetsDir . DIRECTORY_SEPARATOR . $targetFilenameInZip;

                    File::copy($sourcePath, $targetPath);
                    Log::debug("Copied asset '{$asset->original_name}' to {$targetPath} (using unique name: {$targetFilenameInZip}) for website ID: {$this->website->id}");
                }
            }
            // 5. Create a ZIP archive of the temporary site directory
            $safeWebsiteName = Str::slug($websiteName ?: 'untitled-website');
            $zipFilename = "{$safeWebsiteName}_{$this->website->id}_" . time() . ".zip";

            // Define paths for storing the ZIP
            $zipDirectoryPath = "webpilotai_sites_zip/{$this->website->id}"; // Directory for this website's zips
            $fullZipStoragePath = "public/{$zipDirectoryPath}/{$zipFilename}"; // Full path for Storage facade
            $absoluteZipDiskPath = Storage::disk('local')->path($fullZipStoragePath); // Absolute path on disk for ZipArchive

            // Ensure the target directory for the ZIP exists
            File::ensureDirectoryExists(dirname($absoluteZipDiskPath));

            // Create the ZIP archive from the temporary site directory
            $zip = new \ZipArchive();
            if ($zip->open($absoluteZipDiskPath, \ZipArchive::CREATE | \ZipArchive::OVERWRITE) === TRUE) {
                $filesInTempDir = new \RecursiveIteratorIterator(
                    new \RecursiveDirectoryIterator($tempSiteDir, \RecursiveDirectoryIterator::SKIP_DOTS),
                    \RecursiveIteratorIterator::LEAVES_ONLY
                );

                foreach ($filesInTempDir as $name => $file) {
                    if (!$file->isDir()) {
                        $filePath = $file->getRealPath();
                        $relativePath = substr($filePath, strlen($tempSiteDir) + 1); // Path relative to the temp site directory
                        $zip->addFile($filePath, $relativePath);
                    }
                }
                $zip->close();
            } else {
                throw new \Exception("Could not create ZIP archive from temporary directory {$tempSiteDir} for website ID: {$this->website->id} at {$absoluteZipDiskPath}");
            }

            $this->website->update([
                'status' => 'completed',
                // Path relative to public/storage link for asset() helper
                'generated_content_path' => "storage/{$zipDirectoryPath}/{$zipFilename}",
                'last_generated_at' => now(),
                'generation_error' => null,
            ]);
            Log::info("Successfully generated website ID: {$this->website->id}. ZIP archive saved to: {$fullZipStoragePath}");
            WebsiteGenerationSucceeded::dispatch($this->website);

        } catch (Throwable $e) {
            Log::error("Error generating website ID {$this->website->id}: {$e->getMessage()} \n" . $e->getTraceAsString());
            $this->website->update([
                'status' => 'failed',
                'generation_error' => $e->getMessage(),
            ]);
            WebsiteGenerationFailed::dispatch($this->website, $e->getMessage());
        } finally {
            // Clean up the temporary site directory in all cases (success or failure after creation)
            if (isset($tempSiteDir) && File::isDirectory($tempSiteDir)) {
                File::deleteDirectory($tempSiteDir);
                Log::info("Cleaned up temporary directory: {$tempSiteDir}");
            }
        }
    }

    /**
     * Handle a job failure.
    */
    public function failed(Throwable $exception): void
    {
        Log::critical("Generation job permanently failed for website ID {$this->website->id}: {$exception->getMessage()}");
        // Ensure status is updated if not already set by the handle() method's catch block,
        // which can happen for unhandled exceptions or job timeouts.
        if ($this->website->status !== Website::STATUS_FAILED) {
             $this->website->update([
                'status' => Website::STATUS_FAILED,
                'generation_error' => "Generation Job Failed: " . Str::limit($exception->getMessage(), 500), // Limit error message length
            ]);
        }
        // Dispatch event for permanent job failure
        // The third parameter indicates the job failed permanently (not just an error within the handle method)
        WebsiteGenerationFailed::dispatch($this->website, "Generation Job Failed: " . $exception->getMessage(), true);
    }
}
