<?php

namespace Modules\WebPilotAI\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Crypt; // Import Crypt facade
use Illuminate\Validation\Rule;
use App\Models\Setting; // Import the Setting model
use Nwidart\Modules\Facades\Module; // <--- ADD THIS LINE to use the Module facade

class ModuleSettingsController extends Controller
{
    private string $moduleName = 'WebPilotAI';
    private string $moduleAlias = 'webpilotai';

    /**
     * Display the module settings page.
     */
    public function index()
    {
        // Change variable name to match what Blade expects ($settingsSchema)
        $settingsSchema = $this->getSettingsConfig();
        $settingsData = []; // Changed from $currentSettings for consistency with Blade

        foreach ($settingsSchema as $setting) {
            $fullKey = $this->moduleAlias . '.' . $setting['key'];
            // Fetch current value from the database settings table
            $dbSetting = Setting::where('key', $fullKey)->first();

            if ($dbSetting) {
                // Ensure the value is cast to the expected type for the form fields
                $settingsData[$fullKey] = $this->castSettingValue($dbSetting->value, $setting['type']);
            } else {
                // Use default if no setting found in DB
                $settingsData[$fullKey] = $setting['default'] ?? null;
            }
        }

        // Pass settingsSchema and settingsData to the view
        return view('webpilotai::admin.modulesettings.index', compact('settingsSchema', 'settingsData'));
    }

    /**
     * Store the updated module settings.
     */
    public function store(Request $request): RedirectResponse
    {
        $settingsSchema = $this->getSettingsConfig(); // Changed from $settingsConfig
        $rules = [];
        $newData = [];

        foreach ($settingsSchema as $setting) {
            $key = $setting['key'];
            $fullKey = $this->moduleAlias . '.' . $key;
            // Construct the actual input name used in the form (based on Blade logic)
            // Assuming the Blade uses 'webpilotai_setting_key' format
            $inputName = $this->moduleAlias . '_' . \Illuminate\Support\Str::slug($key, '_');

            // Dynamically build validation rules based on 'rules' in module.json
            if (isset($setting['rules'])) {
                $rules[$fullKey] = $setting['rules'];
            }

            // Prepare data for saving
            switch ($setting['type']) {
                case 'boolean':
                case 'checkbox': // Added checkbox type for clarity
                    // Get input using the underscore name from the request
                    $inputValue = $request->input($inputName);
                    // The hidden input ensures $fullKey is always present with '0' or '1'
                    $newData[$fullKey] = $inputValue === '1' ? '1' : '0';
                    break;
                case 'number':
                    $newData[$fullKey] = $request->input($inputName);
                    break;
                case 'password':
                    $inputValue = $request->input($inputName);
                    // If password field is submitted empty, keep the existing value
                    if (empty($inputValue)) {
                        $dbSetting = Setting::where('key', $fullKey)->first();
                        $newData[$fullKey] = $dbSetting ? $dbSetting->value : null;
                    } else {
                        $newData[$fullKey] = Crypt::encryptString($inputValue); // Encrypt the new password
                    }
                    break;
                case 'text':
                case 'textarea':
                case 'select':
                default:
                    $newData[$fullKey] = $request->input($inputName);
                    break;
            }
        }


        try {
            foreach ($settingsSchema as $setting) {
                $fullKey = $this->moduleAlias . '.' . $setting['key'];
                // Get the value from the prepared $newData array
                $valueToStore = $newData[$fullKey];
                Setting::updateOrCreate(
                    ['key' => $fullKey],
                    [
                        'name' => $setting['label'], // Use 'label' from module.json for the 'name' column
                        'value' => $valueToStore, // Value is already processed for type and password logic
                        'type' => $setting['type'],
                        'group' => $setting['group'] ?? $this->moduleAlias,
                    ]
                );
            }

            // It's good practice to clear config cache if settings are being used in config() helper
            Artisan::call('config:clear'); // This will clear all cached config, consider more specific if needed
            Artisan::call('cache:clear'); // Clear general cache
            Artisan::call('view:clear'); // Clear view cache
        } catch (\Exception $e) {
            Log::error("Failed to update WebPilotAI settings: " . $e->getMessage());
            return redirect()->back()->withErrors(['msg' => 'Failed to save settings. An error occurred.']);
        }

        return redirect()->route('admin.webpilotai.settings.index')->with('success', $this->moduleName . ' settings updated successfully!');
    }

    /**
     * Retrieves the settings configuration from module.json.
     */
    private function getSettingsConfig(): array
    {
        // Use the Module facade for more robust module path resolution
        $module = Module::find($this->moduleName);
        if ($module) {
            $moduleConfig = $module->json()->getAttributes();
            return $moduleConfig['settings'] ?? [];
        }

        Log::warning("module.json not found or 'settings' key missing for module: " . $this->moduleName);
        return [];
    }

    /**
     * Casts the setting value from string (database) to its correct type for form display.
     */
    private function castSettingValue(?string $value, string $type)
    {
        if ($value === null) {
            return null;
        }
        switch ($type) {
            case 'boolean':
            case 'checkbox':
                return (bool) $value; // Cast to boolean
            case 'number':
                return is_numeric($value) ? ($value + 0) : $value; // Converts to int/float
            default:
                return $value;
        }
    }

    // No need for prepareValueForStorage as it's handled in the store method now.
    // private function prepareValueForStorage($value, string $type): ?string
    // {
    //     if ($type === 'boolean' || $type === 'checkbox') {
    //         return $value ? '1' : '0';
    //     }
    //     return $value === null ? null : (string) $value;
    // }
}