<?php

namespace Modules\WebPilotAI\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
// use Illuminate\Http\Response; // Not strictly needed if not returning raw Response objects
use Modules\WebPilotAI\Models\AIStylePreset;
use Illuminate\Support\Str; // For slug generation
use Illuminate\Support\Facades\Storage; // For file storage

class AIStylePresetController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $aiStylePresets = AIStylePreset::orderBy('name')->paginate(15); // Changed variable name and order_by
        return view('webpilotai::admin.ai_style_presets.index', compact('aiStylePresets'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $types = [
            'color_palette' => 'Color Palette',
            'font_scheme' => 'Font Scheme',
            'visual_theme' => 'Visual Theme',
            'element_specific_styles' => 'Element Specific Styles',
            'direct_instruction' => 'Direct Instruction'
        ]; // Define available types
        return view('webpilotai::admin.ai_style_presets.create', compact('types'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $validatedData = $request->validate([
            'name' => 'required|string|max:255|unique:ai_style_presets,name',
            'description' => 'nullable|string',
            'type' => 'required|string|in:color_palette,font_scheme,visual_theme,element_specific_styles,direct_instruction',
            'configuration' => ['required', 'string', function ($attribute, $value, $fail) {
                json_decode($value);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    $fail('The ' . $attribute . ' must be a valid JSON string.');
                }
            }],
            'preview_image_path' => 'nullable|image|mimes:jpg,jpeg,png,gif,webp|max:2048',
            'is_active' => 'nullable|boolean', // Changed to nullable for checkbox
            'is_premium' => 'nullable|boolean', // Changed to nullable for checkbox
            // 'sort_order' => 'nullable|integer', // Defaulted below
        ]);

        $data = $validatedData;
        $data['is_active'] = $request->has('is_active');
        $data['is_premium'] = $request->has('is_premium');
        $data['slug'] = Str::slug($validatedData['name']);

        if ($request->hasFile('preview_image_path')) {
            $data['preview_image_path'] = $request->file('preview_image_path')->store('webpilotai/style_previews', 'public');
        }

        $data['configuration'] = json_decode($validatedData['configuration'], true); // Store as array
        $data['sort_order'] = $request->input('sort_order', 0); // Default sort_order if not provided

        AIStylePreset::create($data);

        return redirect()->route('admin.webpilotai.ai-style-presets.index')->with('success', 'AI Style Preset created successfully.');
    }

    /**
     * Show the specified resource.
     */
    public function show(AIStylePreset $aiStylePreset) // Using Route Model Binding
    {
        return redirect()->route('admin.webpilotai.ai-style-presets.edit', $aiStylePreset->id);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(AIStylePreset $aiStylePreset) // Using Route Model Binding
    {
        $types = [
            'color_palette' => 'Color Palette',
            'font_scheme' => 'Font Scheme',
            'visual_theme' => 'Visual Theme',
            'element_specific_styles' => 'Element Specific Styles',
            'direct_instruction' => 'Direct Instruction'
        ];
        // Pass the model instance with the key 'aiStylePreset' to match the view.
        return view('webpilotai::admin.ai_style_presets.edit', ['aiStylePreset' => $aiStylePreset, 'types' => $types]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, AIStylePreset $aiStylePreset): RedirectResponse // Using Route Model Binding
    {
        $validatedData = $request->validate([
            'name' => 'required|string|max:255|unique:ai_style_presets,name,' . $aiStylePreset->id,
            'description' => 'nullable|string',
            'type' => 'required|string|in:color_palette,font_scheme,visual_theme,element_specific_styles,direct_instruction',
            'configuration' => ['required', 'string', function ($attribute, $value, $fail) {
                json_decode($value);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    $fail('The ' . $attribute . ' must be a valid JSON string.');
                }
            }],
            'preview_image_path' => 'nullable|image|mimes:jpg,jpeg,png,gif,webp|max:2048',
            'is_active' => 'nullable|boolean',
            'is_premium' => 'nullable|boolean',
            // 'sort_order' => 'nullable|integer', // Handled below
        ]);

        $data = $validatedData;
        $data['is_active'] = $request->has('is_active');
        $data['is_premium'] = $request->has('is_premium');
        $data['slug'] = Str::slug($validatedData['name']);

        if ($request->hasFile('preview_image_path')) {
            // Delete old image if it exists
            if ($aiStylePreset->preview_image_path) {
                Storage::disk('public')->delete($aiStylePreset->preview_image_path);
            }
            $data['preview_image_path'] = $request->file('preview_image_path')->store('webpilotai/style_previews', 'public');
        } elseif ($request->has('remove_preview_image') && $aiStylePreset->preview_image_path) {
            // Handle removal of preview image
            Storage::disk('public')->delete($aiStylePreset->preview_image_path);
            $data['preview_image_path'] = null;
        }

        $data['configuration'] = json_decode($validatedData['configuration'], true);
        $data['sort_order'] = $request->input('sort_order', $aiStylePreset->sort_order); // Keep existing or update

        $aiStylePreset->update($data);

        return redirect()->route('admin.webpilotai.ai-style-presets.index')->with('success', 'AI Style Preset updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(AIStylePreset $aiStylePreset): RedirectResponse // Using Route Model Binding
    {
        if ($aiStylePreset->preview_image_path) {
            Storage::disk('public')->delete($aiStylePreset->preview_image_path);
        }
        $aiStylePreset->delete();

        return redirect()->route('admin.webpilotai.ai-style-presets.index')->with('success', 'AI Style Preset deleted successfully.');
    }
}
