<?php

namespace Modules\BusinessDirectory\Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Database\Eloquent\Model;
use Modules\BusinessDirectory\Entities\Company;
use Modules\BusinessDirectory\Entities\CompanyType;
use Modules\BusinessDirectory\Entities\Product;
use Modules\BusinessDirectory\Entities\Tender;
use Modules\BusinessDirectory\Entities\Bid;
use Modules\BusinessDirectory\Entities\Job;
use Modules\BusinessDirectory\Entities\JobApplication;
use Modules\BusinessDirectory\Entities\Review;
use Modules\BusinessDirectory\Entities\CompanyProject;
use Modules\BusinessDirectory\Entities\CompanyTask;
use Modules\BusinessDirectory\Entities\Order;
use Modules\BusinessDirectory\Entities\OrderItem;
use Modules\BusinessDirectory\Entities\Cart;
use Modules\BusinessDirectory\Entities\CartItem;
use App\Entities\User; // Assuming global User model

class BusinessDirectoryDatabaseSeeder extends Seeder
{
    /**
     * Run the database seeds.
     *
     * @return void
     */
    public function run()
    {
        Model::unguard();

        // Ensure some users exist or create them
        if (User::count() < 10) {
            User::factory(10)->create();
        }
        $users = User::all();

        // Create Company Types
        $companyTypes = CompanyType::factory(5)->create();
        echo "Company Types seeded.\n";

        // Create Companies, each with some associated data
        Company::factory(20)->create()->each(function ($company) use ($users, $companyTypes) {
            // Assign a random company type if not already set by factory
            if (!$company->company_type_id) {
                $company->company_type_id = $companyTypes->random()->id;
            }
            // Assign a random user as owner if not already set
            if (!$company->user_id) {
                $company->user_id = $users->random()->id;
            }
            $company->save();

            // Create Products for the company
            Product::factory(rand(2, 7))->create(['company_id' => $company->id]);

            // Create Tenders for the company
            Tender::factory(rand(0, 3))->create(['company_id' => $company->id, 'user_id' => $company->user_id])
                ->each(function ($tender) use ($users) {
                    // Create Bids for each tender
                    Bid::factory(rand(1, 5))->create([
                        'tender_id' => $tender->id,
                        'user_id' => $users->random()->id,
                        // Optionally link bid to another company
                        'company_id' => rand(0,1) ? Company::inRandomOrder()->first()->id : null,
                    ]);
                });

            // Create Jobs for the company
            Job::factory(rand(1, 4))->create(['company_id' => $company->id])
                ->each(function ($job) use ($users) {
                    // Create Job Applications for each job
                    JobApplication::factory(rand(0, 8))->create([
                        'job_id' => $job->id,
                        'user_id' => $users->random()->id,
                    ]);
                });

            // Create Reviews for the company
            Review::factory(rand(0, 10))->create([
                'company_id' => $company->id,
                'user_id' => $users->random()->id,
            ]);

            // Create Projects for the company
            CompanyProject::factory(rand(0, 2))->create(['company_id' => $company->id])
                ->each(function ($project) use ($users) {
                    // Create Tasks for each project
                    CompanyTask::factory(rand(3, 10))->create([
                        'project_id' => $project->id,
                        'assignee_user_id' => $users->random()->id,
                    ]);
                });

            // Create Orders (where this company is the seller)
            Order::factory(rand(0,5))->create(['company_id' => $company->id, 'user_id' => $users->random()->id])
                ->each(function($order){
                    OrderItem::factory(rand(1,4))->create(['order_id' => $order->id]);
                    $order->recalculateTotal(); // Recalculate total after items are added
                });
        });
        echo "Companies and their related data seeded.\n";

        // Carts are usually more dynamic, but we can create a few for testing
        Cart::factory(5)->create(['user_id' => $users->random()->id])->each(function($cart){
            CartItem::factory(rand(1,3))->create(['cart_id' => $cart->id]);
        });
        echo "Carts seeded.\n";

        echo "BusinessDirectory module seeded successfully!\n";
    }
}