<?php

namespace Modules\BusinessDirectory\Entities; // Or Modules\BusinessDirectory\Entities if using older nwidart structure

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany; // Import HasMany
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;
use App\Entities\User; // Assuming your User model is in App\Entities
// If CompanyType is in the same namespace:
// use Modules\BusinessDirectory\Entities\CompanyType;
// Or if it's in Entities:
use Modules\BusinessDirectory\Entities\CompanyType as LegacyCompanyType; // Example if old structure
use Modules\BusinessDirectory\Entities\Bid; // Import the Bid model

class Company extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'bd_companies';

    protected $fillable = [
        'user_id',
        'company_type_id',
        'name',
        'slug',
        'description',
        'logo_path',
        'banner_image_path',
        'address_line_1',
        'address_line_2',
        'city',
        'state_province',
        'postal_code',
        'country_code',
        'latitude',
        'longitude',
        'phone_number',
        'email',
        'website_url',
        'whatsapp_number',
        'status',
        'is_featured',
        'is_ecommerce',
    ];

    protected $casts = [
        'is_featured' => 'boolean',
        'is_ecommerce' => 'boolean',
        'latitude' => 'decimal:7',
        'longitude' => 'decimal:7',
    ];

    protected static function newFactory()
    {
        return \Modules\BusinessDirectory\Database\factories\CompanyFactory::new();
    }

    protected static function boot() {
        parent::boot();
        static::creating(function ($company) {
            if (empty($company->slug)) {
                $company->slug = Str::slug($company->name);
            }
            // Ensure unique slug if creating
            $originalSlug = $company->slug;
            $count = 1;
            while (static::where('slug', $company->slug)->exists()) {
                $company->slug = $originalSlug . '-' . $count++;
            }
        });
    }

    public function owner(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function companyType(): BelongsTo
    {
        // Adjust CompanyType import/namespace if necessary
        return $this->belongsTo(LegacyCompanyType::class, 'company_type_id'); // Using alias for clarity
    }

    /**
     * Get all bids made by this company.
     */
    public function bidsMade(): HasMany
    {
        return $this->hasMany(Bid::class, 'bidder_company_id')->orderBy('created_at', 'desc');
    }
}