<?php

namespace Modules\BusinessDirectory\Providers;

use Illuminate\Support\Facades\Blade;
use Illuminate\Foundation\Support\Providers\AuthServiceProvider as ServiceProvider;
use Nwidart\Modules\Traits\PathNamespace;
use Modules\BusinessDirectory\Providers\RouteServiceProvider;

// Entities
use Modules\BusinessDirectory\Entities\Bid;
use Modules\BusinessDirectory\Entities\Company;
use Modules\BusinessDirectory\Entities\CompanyProject;
use Modules\BusinessDirectory\Entities\CompanyTask;
use Modules\BusinessDirectory\Entities\CompanyType;
use Modules\BusinessDirectory\Entities\Job;
use Modules\BusinessDirectory\Entities\Order;
use Modules\BusinessDirectory\Entities\Product;
use Modules\BusinessDirectory\Entities\Review;
use Modules\BusinessDirectory\Entities\Tender;
use Modules\BusinessDirectory\Entities\Category;
use Modules\BusinessDirectory\Settings\BusinessDirectorySettings;
use Modules\BusinessDirectory\Policies\BidPolicy;
use Modules\BusinessDirectory\Policies\CompanyPolicy;
use Modules\BusinessDirectory\Policies\CompanyProjectPolicy;
use Modules\BusinessDirectory\Policies\CompanyTaskPolicy;
use Modules\BusinessDirectory\Policies\CompanyTypePolicy;
use Modules\BusinessDirectory\Policies\JobPolicy;
use Modules\BusinessDirectory\Policies\OrderPolicy;
use Modules\BusinessDirectory\Policies\ProductPolicy;
use Modules\BusinessDirectory\Policies\ReviewPolicy;
use Modules\BusinessDirectory\Policies\TenderPolicy;
use Modules\BusinessDirectory\Policies\BusinessDirectorySettingPolicy;
use Modules\BusinessDirectory\Policies\CategoryPolicy;


use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;


class BusinessDirectoryServiceProvider extends ServiceProvider
{
    use PathNamespace;

    protected string $name = 'BusinessDirectory';

    protected string $nameLower = 'businessdirectory';

    /**
     * The policy mappings for the application.
     *
     * @var array
     */
    protected $policies = [
        Bid::class                          => BidPolicy::class,
        Company::class                      => CompanyPolicy::class,
        CompanyProject::class               => CompanyProjectPolicy::class,
        CompanyTask::class                  => CompanyTaskPolicy::class,
        CompanyType::class                  => CompanyTypePolicy::class,
        Job::class                          => JobPolicy::class,
        Order::class                        => OrderPolicy::class,
        Product::class                      => ProductPolicy::class,
        Review::class                       => ReviewPolicy::class,
        Tender::class                       => TenderPolicy::class,
        BusinessDirectorySettings::class    => BusinessDirectorySettingPolicy::class,
    ];

    /**
     * Boot the application events.
     */
    public function boot(): void
    {
        $this->registerCommands();
        $this->registerPolicies(); // Register policies
        $this->registerCommandSchedules();
        $this->registerTranslations();
        $this->registerConfig();
        $this->registerViews();
        $this->loadMigrationsFrom(module_path($this->name, 'database/migrations'));
    }

    /**
     * Register the service provider.
     */
    public function register(): void
    {
        $this->app->register(EventServiceProvider::class);
        $this->app->register(RouteServiceProvider::class);
    }

    /**
     * Register commands in the format of Command::class
     */
    protected function registerCommands(): void
    {
        // $this->commands([]);
    }

    /**
     * Register command Schedules.
     */
    protected function registerCommandSchedules(): void
    {
        // $this->app->booted(function () {
        //     $schedule = $this->app->make(Schedule::class);
        //     $schedule->command('inspire')->hourly();
        // });
    }

    /**
     * Register translations.
     */
    public function registerTranslations(): void
    {
        $langPath = resource_path('lang/modules/'.$this->nameLower);

        if (is_dir($langPath)) {
            $this->loadTranslationsFrom($langPath, $this->nameLower);
            $this->loadJsonTranslationsFrom($langPath);
        } else {
            $this->loadTranslationsFrom(module_path($this->name, 'lang'), $this->nameLower);
            $this->loadJsonTranslationsFrom(module_path($this->name, 'lang'));
        }
    }

    /**
     * Register config.
     */
    protected function registerConfig(): void
    {
        $configPath = module_path($this->name, config('modules.paths.generator.config.path'));

        if (is_dir($configPath)) {
            $iterator = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($configPath));

            foreach ($iterator as $file) {
                if ($file->isFile() && $file->getExtension() === 'php') {
                    $config = str_replace($configPath.DIRECTORY_SEPARATOR, '', $file->getPathname());
                    $config_key = str_replace([DIRECTORY_SEPARATOR, '.php'], ['.', ''], $config);
                    $segments = explode('.', $this->nameLower.'.'.$config_key);

                    // Remove duplicated adjacent segments
                    $normalized = [];
                    foreach ($segments as $segment) {
                        if (end($normalized) !== $segment) {
                            $normalized[] = $segment;
                        }
                    }

                    $key = ($config === 'config.php') ? $this->nameLower : implode('.', $normalized);

                    $this->publishes([$file->getPathname() => config_path($config)], 'config');
                    $this->merge_config_from($file->getPathname(), $key);
                }
            }
        }
    }

    /**
     * Merge config from the given path recursively.
     */
    protected function merge_config_from(string $path, string $key): void
    {
        $existing = config($key, []);
        $module_config = require $path;

        config([$key => array_replace_recursive($existing, $module_config)]);
    }

    /**
     * Register views.
     */
    public function registerViews(): void
    {
        $viewPath = resource_path('views/modules/'.$this->nameLower);
        $sourcePath = module_path($this->name, 'resources/views');

        $this->publishes([$sourcePath => $viewPath], ['views', $this->nameLower.'-module-views']);

        $this->loadViewsFrom(array_merge($this->getPublishableViewPaths(), [$sourcePath]), $this->nameLower);

        Blade::componentNamespace(config('modules.namespace').'\\' . $this->name . '\\View\\Components', $this->nameLower);
    }

    /**
     * Get the services provided by the provider.
     */
    public function provides(): array
    {
        return [];
    }

    private function getPublishableViewPaths(): array
    {
        $paths = [];
        foreach (config('view.paths') as $path) {
            if (is_dir($path.'/modules/'.$this->nameLower)) {
                $paths[] = $path.'/modules/'.$this->nameLower;
            }
        }

        return $paths;
    }
}
