<?php

namespace Modules\BusinessDirectory\Policies;

use App\Helpers\UserAccessHelper; // Import the helper
use App\Models\User;
use Modules\BusinessDirectory\Entities\Tender;
use Modules\BusinessDirectory\Entities\Company;
use Illuminate\Auth\Access\HandlesAuthorization;

class TenderPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any models.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function viewAny(User $user)
    {
        // Anyone can view the list of tenders (public index page)
        return true;
    }

    /**
     * Determine whether the user can view the model.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Tender  $tender
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function view(?User $user, Tender $tender)
    {
        // Allow viewing if the tender is open, or if the user owns it
        if ($tender->status === 'open') {
            return true;
        }
        // Allow owner or admin to view non-open tenders
        if ($user) {
            return $user->id === $tender->user_id || $user->isAdmin();
        }

        return false;
    }

    /**
     * Determine whether the user can create models.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function create(User $user)
    {
        // 1. Check general feature access for posting tenders
        if (!UserAccessHelper::canAccessFeature($user, 'businessdirectory_post_tenders')) {
            return false;
        }
        // 2. User must have at least one approved company to post a tender for
        return $user->companies()->where('status', 'approved')->exists();
    }

    /**
     * Determine whether the user can create a tender for a specific company.
     * This is used in the store method.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Company  $company
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function createForCompany(User $user, Company $company)
    {
        // 1. Check general feature access for posting tenders
        if (!UserAccessHelper::canAccessFeature($user, 'businessdirectory_post_tenders')) {
            return false;
        }
        // 2. User must own the company and the company must be approved
        return $user->id === $company->user_id && $company->status === 'approved';
    }

    /**
     * Determine whether the user can update the model.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Tender  $tender
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function update(User $user, Tender $tender)
    {
        // 1. Admins can update any tender
        if ($user->isAdmin()) {
            return true;
        }
        // 2. User must own the tender
        if ($user->id !== $tender->user_id) {
            return false;
        }
        // 3. User must have the feature to post (and thus manage) tenders
        return UserAccessHelper::canAccessFeature($user, 'businessdirectory_post_tenders');
    }

    /**
     * Determine whether the user can delete the model.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Tender  $tender
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function delete(User $user, Tender $tender)
    {
        // 1. Admins can delete any tender
        if ($user->isAdmin()) {
            return true;
        }
        // 2. User must own the tender
        if ($user->id !== $tender->user_id) {
            return false;
        }
        // 3. User must have the feature to post (and thus manage) tenders
        return UserAccessHelper::canAccessFeature($user, 'businessdirectory_post_tenders');
    }

    /**
     * Determine whether the user can manage bids for the tender.
     */
    public function manageBids(User $user, Tender $tender): bool
    {
        // 1. Admins can manage bids for any tender
        if ($user->isAdmin()) {
            return true;
        }
        // 2. User must own the tender AND have the feature to post (and thus manage) tenders
        return $user->id === $tender->user_id && UserAccessHelper::canAccessFeature($user, 'businessdirectory_post_tenders');
    }
}