<?php

namespace Modules\BusinessDirectory\Policies;

use App\Helpers\UserAccessHelper; // Import the helper
use App\Models\User; // Assuming your User model path is App\Models\User
use Illuminate\Auth\Access\HandlesAuthorization;
use Modules\BusinessDirectory\Entities\Review;
use Modules\BusinessDirectory\Entities\Company; // Needed to check company

class ReviewPolicy
{
    use HandlesAuthorization;

    /**
     * Perform pre-authorization checks.
     *
     * @param  \App\Models\User  $user
     * @param  string  $ability
     * @return bool|null
     */
    public function before(User $user, string $ability): ?bool
    {
        return $user->isAdmin() ? true : null;
    }

    /**
     * Determine whether the user can view any models.
     *
     * @param  \App\Models\User  $user
     * @return bool
     */
    public function viewAny(User $user): bool
    {
        return $user->isAdmin(); // Admins can view the list of all reviews
    }

    /**
     * Determine whether the user can view the review.
     * Public reviews are viewable by anyone. This is for unapproved reviews.
     */
    public function view(?User $user, Review $review): bool
    {
        // Approved reviews are publicly viewable
        if ($review->is_approved) {
            return true;
        }

        // Allow the user who wrote the review to view it
        if ($user && $review->user_id === $user->id) {
            return true;
        }

        // Allow the owner of the company being reviewed to view it
        if ($user && $review->company && $review->company->user_id === $user->id) {
            return true;
        }

        // Allow administrators to view any review
        if ($user && $user->isAdmin()) { // Use isAdmin() method
            return true;
        }

        return false; // Default deny
    }

    /**
     * Determine whether the user can create reviews for a company.
     * This policy method receives the Review class and the Company instance.
     */
    public function create(User $user, Company $company): bool
    {
        // 1. General feature access for submitting reviews was previously checked here.
        // If "anyone" (authenticated) must be able to submit reviews, this check might be too restrictive.
        // The submission *limit* is handled in the ReviewController using UserAccessHelper::checkLimitAndDecrement.
        // So, we remove the UserAccessHelper::canAccessFeature check from this policy method.

        // 2. User cannot review their own company
        if ($company->user_id === $user->id) {
            return false;
        }

        // 3. User cannot review the same company multiple times (this is also checked in controller, but good for policy too)
        if (Review::where('company_id', $company->id)->where('user_id', $user->id)->exists()) {
            return false;
        }

        // TODO: Add eligibility check (e.g., must have interacted with the company)
        // This might involve checking orders, bids, etc., depending on your business logic.
        // For now, if they have the feature and don't own the company, and haven't reviewed, allow.

        return true;
    }

    /**
     * Determine whether the user can update the review.
     */
    public function update(User $user, Review $review): bool
    {
        // 1. Check general feature access for submitting reviews (implies managing own reviews)
        if (!UserAccessHelper::canAccessFeature($user, 'businessdirectory_submit_reviews')) {
            return false;
        }
        // 2. User must be the author of the review
        return $review->user_id === $user->id;
    }

    /**
     * Determine whether the user can delete the review.
     */
    public function delete(User $user, Review $review): bool
    {
        // Admins can delete any review
        if ($user->isAdmin()) {
            return true;
        }

        // User who wrote the review can delete it, if they have the feature
        if ($review->user_id === $user->id && UserAccessHelper::canAccessFeature($user, 'businessdirectory_submit_reviews')) {
            return true;
        }

        // Owner of the company can delete reviews about their company (if they have company management feature)
        // Assuming 'businessdirectory_list_company' implies ability to manage own company details including reviews on it.
        // Adjust feature key if you have a more specific one for "manage company reviews".
        if ($review->company && $review->company->user_id === $user->id && UserAccessHelper::canAccessFeature($user, 'businessdirectory_list_company')) {
            return true;
        }

        return false;
    }

    // Add policy methods for admin actions like approve/reject
    // public function approve(User $user, Review $review): bool { return $user->is_admin; }
    // public function reject(User $user, Review $review): bool { return $user->is_admin; }
}