<?php

namespace Modules\BusinessDirectory\Policies;
use App\Helpers\UserAccessHelper; // Import the helper

use App\Models\User; // Assuming your User model path is App\Models\User
use Illuminate\Auth\Access\HandlesAuthorization;
use Modules\BusinessDirectory\Entities\Product;
use Modules\BusinessDirectory\Entities\Company; // Needed to check company ownership

class ProductPolicy
{
    use HandlesAuthorization;

    /**
     * Perform pre-authorization checks.
     *
     * @param  \App\Models\User  $user
     * @param  string  $ability
     * @return bool|null
     */
    public function before(User $user, string $ability): ?bool
    {
        return $user->isAdmin() ? true : null;
    }

    /**
     * Determine whether the user can view the product.
     */
    public function view(?User $user, Product $product): bool
    {
        // Published products are publicly viewable
        if ($product->is_published) {
            return true;
        }

        // Allow the owner of the company that owns the product to view it regardless of status
        if ($user && $product->company && $product->company->user_id === $user->id) {
            return true;
        }

        return false; // Default deny
    }

    /**
     * Determine whether the user can create products for a company.
     * This policy method receives the Product class and potentially a Company instance.
     */
    public function create(User $user, ?Company $company = null): bool
    {
        // 1. Check general feature access for managing products
        if (!UserAccessHelper::canAccessFeature($user, 'businessdirectory_manage_products')) {
            return false;
        }

        // 2. If creating within a specific company context:
        if ($company) {
            // User must own that company and it must be approved
            return $company->user_id === $user->id && $company->status === 'approved';
        }

        // 3. If creating generally (e.g., from a global "Post Product" link),
        // user must own at least one approved company to which they can assign the product.
        return $user->companies()->where('status', 'approved')->exists();
    }

    /**
     * Determine whether the user can update the product.
     */
    public function update(User $user, Product $product): bool
    {
        // 1. Check general feature access for managing products
        if (!UserAccessHelper::canAccessFeature($user, 'businessdirectory_manage_products')) {
            return false;
        }
        // 2. User must own the company that owns the product
        return $product->company && $product->company->user_id === $user->id;
    }

    /**
     * Determine whether the user can delete the product.
     */
    public function delete(User $user, Product $product): bool
    {
        // 1. Check general feature access for managing products
        if (!UserAccessHelper::canAccessFeature($user, 'businessdirectory_manage_products')) {
            return false;
        }
        // 2. User must own the company that owns the product
        return $product->company && $product->company->user_id === $user->id;
    }

    // A general 'manage' policy could also be used for create/update/delete if logic is the same
    // public function manage(User $user, Product $product): bool
    // {
    //     return $product->company && $product->company->user_id === $user->id;
    // }
}