<?php

namespace App\Policies;

use App\Models\User;
use Modules\BusinessDirectory\Entities\Order;

class OrderPolicy
{
    /**
     * Perform pre-authorization checks.
     *
     * @param  \App\Models\User  $user
     * @param  string  $ability
     * @return bool|null
     */
    public function before(User $user, string $ability): bool|null
    {
        if ($user->isAdmin()) { // Assuming you have an isAdmin() method on your User model
            return true;
        }
        return null;
    }

    /**
     * Determine whether the user can view any models.
     *
     * @param  \App\Models\User  $user
     * @return bool
     */
    public function viewAny(User $user): bool
    {
        // Admins can view all (handled by before)
        // For frontend, a user might view their own orders.
        return true; // For admin index page
    }

    /**
     * Determine whether the user can view the model.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Order  $order
     * @return bool
     */
    public function view(User $user, Order $order): bool
    {
        // User can view their own order or if they are the owner of the company selling the product.
        return $user->id === $order->user_id || ($order->company && $user->id === $order->company->user_id);
    }

    /**
     * Determine whether the user can create models.
     *
     * @param  \App\Models\User  $user
     * @return bool
     */
    public function create(User $user): bool
    {
        // Typically, users create orders through a checkout process on the frontend.
        // Admins usually don't create orders manually this way.
        return true; // Allow if a user is authenticated (for frontend)
    }

    /**
     * Determine whether the user can update the model.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Order  $order
     * @return bool
     */
    public function update(User $user, Order $order): bool
    {
        // Admins can update order status (handled by before).
        // Users generally cannot update their orders once placed.
        return false;
    }

    /**
     * Determine whether the user can delete the model.
     * Orders are typically not deleted, but cancelled or refunded.
     */
    public function delete(User $user, Order $order): bool
    {
        return false;
    }
}