<?php

namespace Modules\BusinessDirectory\Policies; // Correct namespace
use App\Helpers\UserAccessHelper; // Import the helper

use App\Models\User; // Assuming your User model path is App\Models\User
use Modules\BusinessDirectory\Entities\Company;
use Modules\BusinessDirectory\Entities\Job;
use Illuminate\Auth\Access\HandlesAuthorization;

class JobPolicy
{
    use HandlesAuthorization;

    /**
     * Perform pre-authorization checks.
     *
     * @param  \App\Models\User  $user
     * @param  string  $ability
     * @return bool|null
     */
    public function before(User $user, string $ability): ?bool
    {
        return $user->isAdmin() ? true : null;
    }

    /**
     * Determine whether the user can view any jobs.
     * This is often used for index pages.
     * Public job listings are viewable by anyone.
     * @param  \App\Models\User|null  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function viewAny(?User $user)
    {
        // Anyone can attempt to view job listings (controller will filter by status 'open' for public)
        return true;
    }

    /**
     * Determine whether the user can view the specific job.
     *
     * @param  \App\Models\User|null  $user // Nullable for guests
     * @param  \Modules\BusinessDirectory\Entities\Job  $job
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function view(?User $user, Job $job)
    {
        // If the job is open, anyone can view it.
        if ($job->status === 'open') {
            return true;
        }

        // Allow authenticated users who own the company or posted the job to view non-open jobs (e.g., drafts).
        if ($user) {
            // Check if user owns the job OR if the job has a company and the user owns that company
            $isOwner = $user->id === $job->user_id || ($job->company && $user->id === $job->company->user_id);
            // Admins can also view non-open jobs
            return $isOwner || $user->isAdmin();
        }

        return false;
    }

    /**
     * Determine whether the user can create jobs.
     * If $company is provided, it checks if the user can create a job for that specific company.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Company  $company
     * @return \Illuminate\Auth\Access\Response|bool
     * This policy method receives the Job class and the Company instance.
     */
    public function create(User $user, Company $company): bool
    { // Note: The controller passes the Company instance, so this method is for creating a job FOR a company.
        // 1. Check general feature access for managing jobs
        if (!UserAccessHelper::canAccessFeature($user, 'businessdirectory_post_jobs')) {
            return false;
        }

        // 2. User must own the company and it must be approved
        return $user->id === $company->user_id && $company->status === 'approved';
    }
    // If you had a general "create job" page where user selects one of their companies,
    // you'd need a separate policy method like:
    // public function createGeneral(User $user): bool {
    //     return UserAccessHelper::canAccessFeature($user, 'businessdirectory_post_jobs') && $user->companies()->where('status', 'approved')->exists();
    // }

    /**
     * Determine whether the user can update the job.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Job  $job
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function update(User $user, Job $job): bool
    {
        // 1. Check general feature access for managing jobs
        if (!UserAccessHelper::canAccessFeature($user, 'businessdirectory_post_jobs')) {
            return false;
        }
        // 2. User who posted the job or owns the company can update it.
        return $user->id === $job->user_id || ($job->company && $user->id === $job->company->user_id);
    }

    /**
     * Determine whether the user can delete the job.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Job  $job
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function delete(User $user, Job $job): bool
    {
        // 1. Check general feature access for managing jobs
        if (!UserAccessHelper::canAccessFeature($user, 'businessdirectory_post_jobs')) {
            return false;
        }
        // 2. User who posted the job or owns the company can delete it.
        return $user->id === $job->user_id || ($job->company && $user->id === $job->company->user_id);
    }

    /**
     * Determine whether the user can manage the job (general permission, often used for edit/delete).
     * Your controller uses 'manage'.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Job  $job
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function manage(User $user, Job $job): bool
    {
        return $this->update($user, $job); // Delegate to update or define specific logic
    }

    /**
     * Determine whether the user can apply for the job.
     */
    public function apply(User $user, Job $job): bool
    {
        // 1. Check feature access for applying to jobs
        if (!UserAccessHelper::canAccessFeature($user, 'businessdirectory_apply_for_jobs')) {
            return false;
        }

        // 2. Job must be open
        if ($job->status !== 'open') {
            return false;
        }
        // 3. User cannot apply to their own job (or their company's job)
        return !($user->id === $job->user_id || ($job->company && $user->id === $job->company->user_id));
    }
}
