<?php

namespace Modules\BusinessDirectory\Policies;

use App\Models\User; // Assuming your User model path is App\Models\User
use Illuminate\Auth\Access\HandlesAuthorization;
use Modules\BusinessDirectory\Entities\CompanyTask;
use Modules\BusinessDirectory\Entities\CompanyProject; // Needed to check project ownership

class CompanyTaskPolicy
{
    use HandlesAuthorization;

    /**
     * Perform pre-authorization checks.
     *
     * @param  \App\Models\User  $user
     * @param  string  $ability
     * @return bool|null
     */
    public function before(User $user, string $ability): ?bool
    {
        return $user->isAdmin() ? true : null;
    }

    /**
     * Determine whether the user can view the company task.
     * This is typically restricted to the company owner or assigned user.
     */
    public function view(User $user, CompanyTask $task): bool
    {
        // User must own the company the project belongs to
        if ($task->project && $task->project->company && $task->project->company->user_id === $user->id) {
            return true;
        }

        // User is the assignee of the task
        if ($task->assignee_user_id === $user->id) {
            return true;
        }

        return false;
    }

    /**
     * Determine whether the user can create company tasks within a project.
     * This policy method receives the CompanyTask class and the parent CompanyProject instance.
     */
    public function create(User $user, CompanyProject $project): bool
    {
        // User must own the company the project belongs to
        return $project->company && $project->company->user_id === $user->id;
    }

    /**
     * Determine whether the user can update the company task.
     */
    public function update(User $user, CompanyTask $task): bool
    {
        // User must own the company the project belongs to
        if ($task->project && $task->project->company && $task->project->company->user_id === $user->id) {
            return true;
        }

        // Optionally, allow the assignee to update status/progress (but maybe not title/description)
        // This would require more granular policy checks or separate methods.
        // For now, let's stick to company owner can update everything.

        return false;
    }

    /**
     * Determine whether the user can delete the company task.
     */
    public function delete(User $user, CompanyTask $task): bool
    {
        // User must own the company the project belongs to
        return $task->project && $task->project->company && $task->project->company->user_id === $user->id;
    }

    // A general 'manage' policy could also be used for create/update/delete if logic is the same
    // public function manage(User $user, CompanyTask $task): bool
    // {
    //     return $task->project && $task->project->company && $task->project->company->user_id === $user->id;
    // }
}