<?php

namespace Modules\BusinessDirectory\Policies;

use App\Helpers\UserAccessHelper; // Import the helper
use App\Models\User;
use Modules\BusinessDirectory\Entities\Company;
use Modules\BusinessDirectory\Entities\CompanyProject;
use Illuminate\Auth\Access\HandlesAuthorization;

class CompanyProjectPolicy
{
    use HandlesAuthorization;

    /**
     * Perform pre-authorization checks.
     *
     * @param  \App\Models\User  $user
     * @param  string  $ability
     * @return bool|null
     */
    public function before(User $user, string $ability): ?bool
    {
        return $user->isAdmin() ? true : null;
    }

    /**
     * Determine whether the user can view any models.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Company  $company
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function viewAny(User $user, Company $company)
    {
        // 1. Check general feature access for managing company projects
        if (!UserAccessHelper::canAccessFeature($user, 'businessdirectory_manage_projects')) {
            return false;
        }
        // 2. User must own the company or be an admin
        return $user->id === $company->user_id;
    }

    /**
     * Determine whether the user can view the model.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\CompanyProject  $companyProject
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function view(User $user, CompanyProject $companyProject)
    {
        // 1. Check general feature access for managing company projects
        if (!UserAccessHelper::canAccessFeature($user, 'businessdirectory_manage_projects')) {
            return false;
        }
        // 2. User must own the company the project belongs to, or be an admin
        return ($companyProject->company && $user->id === $companyProject->company->user_id);
    }

    /**
     * Determine whether the user can create models.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Company  $company The company for which the project is being created
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function create(User $user, Company $company)
    {
        // 1. Check general feature access for managing company projects
        if (!UserAccessHelper::canAccessFeature($user, 'businessdirectory_manage_projects')) {
            return false;
        }
        // 2. User must own the company and company must be approved
        return $user->id === $company->user_id && $company->status === 'approved';
    }

    /**
     * Determine whether the user can update the model.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\CompanyProject  $companyProject
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function update(User $user, CompanyProject $companyProject)
    {
        // 1. Check general feature access for managing company projects
        if (!UserAccessHelper::canAccessFeature($user, 'businessdirectory_manage_projects')) {
            return false;
        }
        // 2. User must own the company the project belongs to, or be an admin
        return ($companyProject->company && $user->id === $companyProject->company->user_id);
    }

    /**
     * Determine whether the user can delete the model.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\CompanyProject  $companyProject
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function delete(User $user, CompanyProject $companyProject)
    {
        // 1. Check general feature access for managing company projects
        if (!UserAccessHelper::canAccessFeature($user, 'businessdirectory_manage_projects')) {
            return false;
        }
        // 2. User must own the company the project belongs to, or be an admin
        return ($companyProject->company && $user->id === $companyProject->company->user_id);
    }

    /**
     * Determine whether the user can manage the model (e.g., add tasks, etc.).
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\CompanyProject  $companyProject
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function manage(User $user, CompanyProject $companyProject)
    {
        // 1. Check general feature access for managing company projects
        if (!UserAccessHelper::canAccessFeature($user, 'businessdirectory_manage_projects')) {
            return false;
        }
        // 2. User must own the company the project belongs to, or be an admin
        return ($companyProject->company && $user->id === $companyProject->company->user_id);
    }

    /**
     * Determine whether the user can restore the model.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\CompanyProject  $companyProject
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function restore(User $user, CompanyProject $companyProject)
    {
        // Admins might be able to restore, but typically not a frontend user action.
        // Add feature check if this becomes a user-facing feature.
        // This will be caught by before() for admins.
        return false; // Non-admins typically cannot restore.
    }

    /**
     * Determine whether the user can permanently delete the model.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\CompanyProject  $companyProject
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function forceDelete(User $user, CompanyProject $companyProject)
    {
        // Admins might be able to force delete, but typically not a frontend user action.
        // Add feature check if this becomes a user-facing feature.
        // This will be caught by before() for admins.
        return false; // Non-admins typically cannot force delete.
    }
}
