<?php

namespace Modules\BusinessDirectory\Policies; // Correct namespace
use App\Helpers\UserAccessHelper; // Import the helper
use App\Models\User; // Assuming your User model path is App\Models\User
use Modules\BusinessDirectory\Entities\Company;
use Illuminate\Auth\Access\HandlesAuthorization;

class CompanyPolicy
{
    use HandlesAuthorization;

    /**
     * Perform pre-authorization checks.
     *
     * @param  \App\Models\User  $user
     * @param  string  $ability
     * @return bool|null
     */
    public function before(User $user, string $ability): ?bool
    {
        return $user->isAdmin() ? true : null;
    }

    /**
     * Determine whether the user can view any Entities.
     *
     * @param  \App\Models\User|null  $user // Nullable for guest access
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function viewAny(?User $user)
    {
        // Anyone can view the list of companies (public directory)
        return true;
    }

    /**
     * Determine whether the user can view the model.
     *
     * @param  \App\Models\User|null  $user // Nullable for guest access
     * @param  \Modules\BusinessDirectory\Entities\Company  $company
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function view(?User $user, Company $company)
    {
        // Approved companies can be viewed by anyone.
        // Pending/rejected/suspended companies might only be viewable by owner or admin.
        if ($company->status === 'approved') {
            return true;
        }

        // Allow owner or admin to view non-approved companies
        if ($user) {
            return $user->id === $company->user_id || $user->isAdmin();
        }

        return false;
    }

    /**
     * Determine whether the user can create Entities.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function create(User $user): bool
    {
        // 1. Check general feature access for listing/creating companies
        if (!UserAccessHelper::canAccessFeature($user, 'businessdirectory_list_company')) {
            return false;
        }
        // 2. Any authenticated user with the feature can attempt to create.
        // Additional checks like max companies per plan will be handled by UserAccessHelper.
        return true;
    }

    /**
     * Determine whether the user can update the model.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Company  $company
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function update(User $user, Company $company): bool
    {
        // 2. User must own the company AND have the feature to manage companies
        return $user->id === $company->user_id && UserAccessHelper::canAccessFeature($user, 'businessdirectory_list_company');
    }

    /**
     * Determine whether the user can delete the model.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Company  $company
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function delete(User $user, Company $company): bool
    {
        // 2. User must own the company AND have the feature to manage companies
        return $user->id === $company->user_id && UserAccessHelper::canAccessFeature($user, 'businessdirectory_list_company');
    }

    /**
     * Determine whether the user can restore the model.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Company  $company
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function restore(User $user, Company $company): bool
    {
        // This will be caught by before() for admins.
        return false; // Non-admins typically cannot restore.
    }

    /**
     * Determine whether the user can permanently delete the model.
     *
     * @param  \App\Models\User  $user
     * @param  \Modules\BusinessDirectory\Entities\Company  $company
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function forceDelete(User $user, Company $company): bool
    {
        // This will be caught by before() for admins.
        return false; // Non-admins typically cannot force delete.
    }
}
