<?php

namespace Modules\BusinessDirectory\Policies;

use App\Models\User; // Assuming your User model path
use App\Helpers\UserAccessHelper; // For checking feature access
use Illuminate\Auth\Access\HandlesAuthorization;
use Modules\BusinessDirectory\Entities\Bid;
use Modules\BusinessDirectory\Entities\Tender; // Import Tender model

class BidPolicy
{
    use HandlesAuthorization;

    /**
     * Perform pre-authorization checks.
     *
     * @param  \App\Models\User  $user
     * @param  string  $ability
     * @return bool|null
     */
    public function before(User $user, string $ability): ?bool
    {
        return $user->isAdmin() ? true : null;
    }

    /**
     * Determine whether the user can view the bid.
     */
    public function view(User $user, Bid $bid): bool
    {
        // User can view the bid if they are the bidder (direct user)
        if ($bid->bidder_user_id === $user->id) {
            return true;
        }

        // User can view the bid if they own the company that submitted the bid
        if ($bid->bidderCompany && $bid->bidderCompany->user_id === $user->id) {
             return true;
        }

        // User can view the bid if they own the company associated with the tender
        if ($bid->tender && $bid->tender->company && $bid->tender->company->user_id === $user->id) {
             return true;
        }

        return false;
    }

    /**
     * Determine whether the user can create bids for a specific tender.
     */
    public function create(User $user, Tender $tender): bool
    {
        // 1. Check general feature access for bidding
        if (!UserAccessHelper::canAccessFeature($user, 'businessdirectory_bid_on_tenders')) {
            return false;
        }

        // 2. Tender must be open for bidding
        if ($tender->status !== 'open') {
            return false;
        }

        // 3. Bidding deadline must not have passed (if set)
        if ($tender->deadline_for_bids && $tender->deadline_for_bids->isPast()) {
            return false;
        }

        // 4. User (or their company) cannot bid on their own tender
        // Check if the tender was posted by the user directly
        if ($tender->user_id === $user->id) {
            return false;
        }
        // Check if the tender was posted by a company owned by the user
        if ($tender->company && $tender->company->user_id === $user->id) {
            return false;
        }

        // TODO: Consider adding a check to prevent multiple bids from the same user/company on the same tender,
        // though this might be better handled in the controller/service layer for specific feedback.

        return true;
    }

    /**
     * Determine whether the user can withdraw the bid.
     */
    public function withdraw(User $user, Bid $bid): bool
    {
        // 1. User must be the bidder (direct user or owner of the bidding company)
        $isBidder = ($bid->bidder_user_id === $user->id) || ($bid->bidderCompany && $bid->bidderCompany->user_id === $user->id);
        if (!$isBidder) {
            return false;
        }

        // 2. Can only withdraw if bid status is 'pending' (or your equivalent initial status)
        if ($bid->status !== 'pending') {
            return false;
        }

        // 3. Tender must still be 'open' and deadline not passed
        if ($bid->tender->status !== 'open' || ($bid->tender->deadline_for_bids && $bid->tender->deadline_for_bids->isPast())) {
            return false;
        }

        return true;
    }

    // Add other policy methods like 'create', 'update', 'delete' if needed
}
