<?php

namespace Modules\BusinessDirectory\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Notification;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Modules\BusinessDirectory\Entities\Tender;
use Modules\BusinessDirectory\Entities\Bid; // If you need bid context
use Illuminate\Support\Str;

class TenderStatusChanged extends Notification implements ShouldQueue
{
    use Queueable;

    public Tender $tender;
    public string $newStatus;
    public ?Bid $relatedBid; // The user's bid, if relevant

    /**
     * Create a new notification instance.
     *
     * @param Tender $tender
     * @param string $newStatus
     * @param Bid|null $relatedBid The specific bid of the user being notified
     */
    public function __construct(Tender $tender, string $newStatus, Bid $relatedBid = null)
    {
        $this->tender = $tender;
        $this->newStatus = $newStatus;
        $this->relatedBid = $relatedBid;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function via($notifiable)
    {
            return ['database', 'mail'];
    }

    /**
     * Get the array representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function toArray($notifiable)
    {
        $message = "The tender \"{$this->tender->title}\"";

        switch ($this->newStatus) {
            case 'awarded':
                if ($this->relatedBid && $this->relatedBid->status === 'accepted') {
                    // This case should be handled by BidStatusChanged, but as a fallback:
                    $message .= " has been awarded, and your bid was accepted!";
                } else {
                    $message .= " has been awarded to another bidder.";
                }
                break;
            case 'closed':
                $message .= " has been closed and is no longer accepting bids.";
                break;
            case 'cancelled':
                $message .= " has been cancelled.";
                break;
            default:
                $message .= " status has changed to " . Str::title(str_replace('_', ' ', $this->newStatus)) . ".";
        }

        return [
            'tender_id' => $this->tender->id,
            'tender_title' => $this->tender->title,
            'tender_slug' => $this->tender->slug,
            'new_status' => $this->newStatus,
            'message' => $message,
            'link' => route('public.businessdirectory.tenders.show', $this->tender->slug),
        ];
        }

        /**
         * Get the mail representation of the notification.
         *
         * @param  mixed  $notifiable
         * @return \Illuminate\Notifications\Messages\MailMessage
         */
        public function toMail($notifiable)
        {
            $url = route('public.businessdirectory.tenders.show', $this->tender->slug);
            $line = $this->toArray($notifiable)['message']; // Reuse the generated message

            return (new MailMessage)
                        ->subject('Update on Tender: ' . Str::limit($this->tender->title, 40))
                        ->greeting('Hello ' . $notifiable->name . ',')
                        ->line($line)
                        ->action('View Tender', $url);
    }
}
