<?php

namespace Modules\BusinessDirectory\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Notification;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Modules\BusinessDirectory\Entities\Bid;
use Modules\BusinessDirectory\Entities\Tender;
use Illuminate\Support\Str; // Add Str facade

class NewBidSubmitted extends Notification implements ShouldQueue // Optional: implements ShouldQueue for background processing
{
    use Queueable;

    public Bid $bid;
    public Tender $tender;

    /**
     * Create a new notification instance.
     *
     * @param Bid $bid The bid that was submitted
     */
    public function __construct(Bid $bid)
    {
        $this->bid = $bid;
        $this->tender = $bid->tender; // Assuming Bid model has a tender relationship
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function via($notifiable)
    {
        return ['database', 'mail']; // Enable mail channel
    }

    /**
     * Get the mail representation of the notification.
     * (Optional - if you add 'mail' to via())
     *
     * @param  mixed  $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toMail($notifiable)
    {
        $url = route('public.businessdirectory.tenders.show', $this->tender->slug) . '#bid-' . $this->bid->id; // Link to the tender and specific bid

        return (new MailMessage)
                    ->subject('New Bid Submitted on Your Tender: ' . Str::limit($this->tender->title, 50))
                    ->greeting('Hello ' . $notifiable->name . ',')
                    ->line("A new bid has been submitted by {$this->bid->bidderCompany->name} for your tender: \"{$this->tender->title}\".")
                    ->lineIf($this->bid->bid_amount, "Bid Amount: " . number_format($this->bid->bid_amount, 2) . " " . ($this->tender->budget_currency ?? 'USD'))
                    ->action('View Bid Details', $url)
                    ->line('Thank you for using our platform!');
    }

    /**
     * Get the array representation of the notification.
     * This is what will be stored in the 'data' column of the 'notifications' table.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function toArray($notifiable)
    {
        return [
            'tender_id' => $this->tender->id,
            'tender_title' => $this->tender->title,
            'tender_slug' => $this->tender->slug,
            'bid_id' => $this->bid->id,
            'bidder_company_id' => $this->bid->bidder_company_id,
            'bidder_company_name' => $this->bid->bidderCompany->name, // Assuming Bid model has bidderCompany relationship
            'message' => "A new bid from {$this->bid->bidderCompany->name} has been submitted for your tender: \"{$this->tender->title}\".",
            'link' => route('public.businessdirectory.tenders.show', $this->tender->slug) . '#bid-' . $this->bid->id, // Link directly to the bid on the tender page
        ];
    }
}
